<?php
/**
 * Send Response to Bug Report or Feature Request
 * API برای ارسال پاسخ به گزارش باگ یا درخواست ویژگی
 */

header('Content-Type: application/json; charset=utf-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config/database.php';
require_once '../includes/functions.php';
require_once '../config/email-config.php';

// Check authentication
if (!isLoggedIn()) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

try {
    $rawInput = file_get_contents('php://input');
    if (empty($rawInput)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'داده‌های درخواست خالی است']);
        exit();
    }
    
    $input = json_decode($rawInput, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'خطا در پردازش JSON: ' . json_last_error_msg()]);
        exit();
    }
    
    $formType = sanitizeInput($input['form_type'] ?? '');
    $id = (int)($input['id'] ?? 0);
    $response = sanitizeInput($input['response'] ?? '');
    
    if (empty($formType) || $id <= 0 || empty($response)) {
        http_response_code(400);
        echo json_encode([
            'success' => false, 
            'message' => 'تمام فیلدها الزامی هستند',
            'debug' => ['form_type' => $formType, 'id' => $id, 'response_length' => strlen($response)]
        ]);
        exit();
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get the submission details
    if ($formType === 'bug_report') {
        $table = 'bug_reports';
        $query = "SELECT id, name, email, subject, description, priority FROM bug_reports WHERE id = ?";
    } else {
        $table = 'feature_requests';
        $query = "SELECT id, name, email, feature as subject, description, priority FROM feature_requests WHERE id = ?";
    }
    
    $stmt = $conn->prepare($query);
    $stmt->execute([$id]);
    $submission = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$submission) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'گزارش یافت نشد']);
        exit();
    }
    
    // Check if columns exist before updating
    $checkColumnsQuery = $formType === 'bug_report' 
        ? "SHOW COLUMNS FROM bug_reports LIKE 'admin_response'"
        : "SHOW COLUMNS FROM feature_requests LIKE 'admin_response'";
    $checkStmt = $conn->query($checkColumnsQuery);
    $hasResponseColumns = $checkStmt->rowCount() > 0;
    $checkStmt->closeCursor();
    
    // Update the submission with admin response
    if ($hasResponseColumns) {
        if ($formType === 'bug_report') {
            $updateQuery = "UPDATE bug_reports SET admin_response = ?, response_sent_at = NOW(), updated_at = NOW() WHERE id = ?";
        } else {
            $updateQuery = "UPDATE feature_requests SET admin_response = ?, response_sent_at = NOW(), updated_at = NOW() WHERE id = ?";
        }
        
        $updateStmt = $conn->prepare($updateQuery);
        $updateStmt->execute([$response, $id]);
    } else {
        // Columns don't exist, just log the response (for backward compatibility)
        error_log("Response columns not found for {$formType} ID: {$id}. Response: " . substr($response, 0, 100));
    }
    
    // Send email to user
    $emailSent = false;
    try {
        $emailSubject = $formType === 'bug_report' 
            ? '📧 پاسخ به گزارش مشکل شما - گواهیتو'
            : '📧 پاسخ به درخواست ویژگی شما - گواهیتو';
        
        $emailBody = sendResponseEmailTemplate($formType, $submission, $response);
        
        $emailSent = sendEmail($submission['email'], $emailSubject, $emailBody);
    } catch (Exception $emailError) {
        error_log("Email sending error: " . $emailError->getMessage());
        // Continue even if email fails
    }
    
    // Log the action
    logSecurityEvent('admin_response_sent', 'Type: ' . $formType . ', ID: ' . $id . ', Email: ' . $submission['email'], 'low');
    
    echo json_encode([
        'success' => true,
        'message' => 'پاسخ با موفقیت ارسال شد',
        'email_sent' => $emailSent
    ]);
    
} catch (PDOException $e) {
    error_log("Send response PDO error: " . $e->getMessage() . " | Code: " . $e->getCode());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'خطا در ارسال پاسخ',
        'error' => $e->getMessage()
    ]);
} catch (Exception $e) {
    error_log("Send response error: " . $e->getMessage() . " | Trace: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'خطا در ارسال پاسخ: ' . $e->getMessage()
    ]);
}

/**
 * Generate email template for admin response
 */
function sendResponseEmailTemplate($formType, $submission, $response) {
    $formTitle = $formType === 'bug_report' ? 'گزارش مشکل' : 'درخواست ویژگی';
    $priorityColor = getPriorityColor($submission['priority']);
    $priorityLabel = getPriorityLabel($submission['priority']);
    
    return '
    <h3 style="color: #041c3f; margin-bottom: 20px;">📧 پاسخ به ' . $formTitle . ' شما</h3>
    
    <div style="margin-bottom: 20px; padding: 15px; background: rgba(4, 28, 63, 0.05); border-right: 4px solid #041c3f; border-radius: 8px;">
        <div class="info-row">
            <span class="info-label">👤 نام:</span>
            <span class="info-value">' . htmlspecialchars($submission['name'], ENT_QUOTES, 'UTF-8') . '</span>
        </div>
        
        <div class="info-row">
            <span class="info-label">📝 ' . ($formType === 'bug_report' ? 'موضوع مشکل' : 'ویژگی درخواستی') . ':</span>
            <span class="info-value">' . htmlspecialchars($submission['subject'], ENT_QUOTES, 'UTF-8') . '</span>
        </div>
        
        <div class="info-row">
            <span class="info-label">⚡ اولویت:</span>
            <span class="info-value">
                <span class="priority-badge" style="background-color: ' . $priorityColor . '">' . $priorityLabel . '</span>
            </span>
        </div>
    </div>
    
    <div style="margin-top: 30px; padding: 20px; background: linear-gradient(135deg, rgba(4, 28, 63, 0.1), rgba(197, 145, 44, 0.1)); border-radius: 12px; border: 2px solid rgba(197, 145, 44, 0.3);">
        <h4 style="color: #041c3f; margin-bottom: 15px; font-size: 18px; font-weight: 700;">💬 پاسخ تیم پشتیبانی:</h4>
        <div style="color: #333; line-height: 1.8; white-space: pre-wrap;">' . nl2br(htmlspecialchars($response, ENT_QUOTES, 'UTF-8')) . '</div>
    </div>
    
    <div style="margin-top: 30px; padding: 15px; background: #e8f5e8; border: 1px solid #4caf50; border-radius: 8px;">
        <p style="margin: 0; color: #2e7d32; font-weight: 600;">✅ این پاسخ به ' . $formTitle . ' شما ارسال شده است</p>
    </div>
    
    <div style="margin-top: 20px; padding: 15px; background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px;">
        <p style="margin: 0; color: #856404;">💡 در صورت نیاز به اطلاعات بیشتر، می‌توانید با ما تماس بگیرید</p>
    </div>';
}
?>

