<?php
/**
 * API Endpoint for Form Submissions
 * API برای ارسال فرم‌های گزارش مشکل و درخواست ویژگی
 */

// Start output buffering to prevent any output before headers
ob_start();

// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Enable CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json; charset=utf-8');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Include required files
require_once '../config/database.php';
require_once '../includes/functions.php';
require_once '../config/email-config.php';
require_once '../config/recaptcha-config.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid JSON input']);
    exit();
}

// Validate required fields
$requiredFields = ['form_type', 'name', 'email', 'subject', 'description', 'priority'];
foreach ($requiredFields as $field) {
    if (empty($input[$field])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => "Field '$field' is required"]);
        exit();
    }
}

// Sanitize input with enhanced security
$formType = preg_replace('/[^a-zA-Z_]/', '', $input['form_type']);
// Allow Persian/Farsi characters in name (keep Unicode letters, numbers, spaces, and common punctuation)
$name = preg_replace('/[<>"\']/', '', $input['name']); // Only remove dangerous HTML chars, keep Persian
$email = filter_var($input['email'], FILTER_SANITIZE_EMAIL);
// Allow Persian/Farsi characters in subject (only remove dangerous HTML chars)
$subject = preg_replace('/[<>"\']/', '', $input['subject']); // Only remove dangerous HTML chars, keep Persian
// Allow Persian/Farsi characters in description (only remove dangerous HTML chars)
$description = preg_replace('/[<>"\']/', '', $input['description']); // Only remove dangerous HTML chars, keep Persian
$priority = preg_replace('/[^a-zA-Z]/', '', $input['priority']);

// Validate form type
if (!in_array($formType, ['bug_report', 'feature_request'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid form type']);
    exit();
}

// Validate email
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid email format']);
    exit();
}

// Validate priority
$validPriorities = ['low', 'medium', 'high', 'critical'];
if (!in_array($priority, $validPriorities)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid priority level']);
    exit();
}

// Security checks
$clientIP = getClientIP();

// Check if IP is blacklisted
if (isIPBlacklisted($clientIP)) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Access denied', 'blocked' => true]);
    logSecurityEvent('form_submission_blocked_blacklisted_ip', 'IP: ' . $clientIP, 'high');
    exit();
}

// Check for excessive form submissions (more than 4 times)
try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Count form submissions in last 1 hour (bug reports + feature requests)
    // This counts previous submissions BEFORE the current one
    $countQuery = "
        SELECT COUNT(*) as submission_count 
        FROM (
            SELECT ip_address, created_at FROM bug_reports WHERE ip_address = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
            UNION ALL
            SELECT ip_address, created_at FROM feature_requests WHERE ip_address = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ) as combined
    ";
    $countStmt = $conn->prepare($countQuery);
    $countStmt->execute([$clientIP, $clientIP]);
    $result = $countStmt->fetch(PDO::FETCH_ASSOC);
    $previousSubmissionCount = (int)($result['submission_count'] ?? 0);
    
    // If 3 or more previous submissions, this will be the 4th, so block the IP
    // This means: 1st, 2nd, 3rd submissions are allowed, 4th submission triggers block
    if ($previousSubmissionCount >= 3) {
        // Get number of previous blocks for this IP
        $blockCountQuery = "SELECT COUNT(*) as block_count FROM ip_blacklist WHERE ip_address = ?";
        $blockCountStmt = $conn->prepare($blockCountQuery);
        $blockCountStmt->execute([$clientIP]);
        $blockResult = $blockCountStmt->fetch(PDO::FETCH_ASSOC);
        $previousBlockCount = (int)($blockResult['block_count'] ?? 0);
        
        // Determine block duration and reason
        $isPermanent = ($previousBlockCount >= 3);
        $expiresAt = $isPermanent ? null : date('Y-m-d H:i:s', strtotime('+3 hours'));
        $reason = $isPermanent 
            ? "ارسال بیش از حد فرم‌ها (بیش از 4 بار) - مسدودسازی دائمی به دلیل تکرار مسدودسازی"
            : "ارسال بیش از حد فرم‌ها (بیش از 4 بار) - مسدودسازی موقت 3 ساعته";
        
        // Get admin user ID for foreign key
        $adminQuery = "SELECT id FROM admin_users LIMIT 1";
        $adminStmt = $conn->query($adminQuery);
        $adminUser = $adminStmt->fetch(PDO::FETCH_ASSOC);
        $adminId = $adminUser ? $adminUser['id'] : 1;
        
        // Block the IP
        $blockQuery = "
            INSERT INTO ip_blacklist (ip_address, reason, expires_at, added_by, created_at, updated_at) 
            VALUES (?, ?, ?, ?, NOW(), NOW())
            ON DUPLICATE KEY UPDATE 
            reason = VALUES(reason), 
            expires_at = VALUES(expires_at), 
            updated_at = NOW()
        ";
        $blockStmt = $conn->prepare($blockQuery);
        $blockStmt->execute([$clientIP, $reason, $expiresAt, $adminId]);
        
        // Log security event with high severity
        $totalSubmissions = $previousSubmissionCount + 1; // Current submission will be the 4th
        logSecurityEvent('form_submission_excessive_blocked', 
            "IP blocked due to excessive form submissions: {$totalSubmissions} submissions (4th submission), Block #{$previousBlockCount}", 
            'critical',
            json_encode([
                'ip' => $clientIP,
                'previous_submissions' => $previousSubmissionCount,
                'total_submissions' => $totalSubmissions,
                'previous_blocks' => $previousBlockCount,
                'is_permanent' => $isPermanent,
                'expires_at' => $expiresAt
            ])
        );
        
        // Return blocked response - don't save the form submission
        http_response_code(403);
        ob_clean();
        echo json_encode([
            'success' => false, 
            'message' => 'به دلیل ارسال بیش از حد فرم‌ها (بیش از 4 بار)، دسترسی شما مسدود شده است.',
            'blocked' => true,
            'reason' => $reason,
            'expires_at' => $expiresAt,
            'is_permanent' => $isPermanent,
            'submission_count' => $totalSubmissions
        ], JSON_UNESCAPED_UNICODE);
        ob_end_flush();
        exit();
    }
} catch (Exception $e) {
    error_log("Error checking form submission count: " . $e->getMessage());
    // Continue with normal flow if check fails
}

// Rate limiting
if (!checkRateLimit('form_submission', 10)) {
    http_response_code(429);
    echo json_encode(['success' => false, 'message' => 'Too many submissions. Please try again later.']);
    logSecurityEvent('form_submission_rate_limit_exceeded', 'IP: ' . $clientIP, 'medium');
    exit();
}

// Verify reCAPTCHA (only if enabled)
$recaptchaEnabled = getSystemSetting('enable_recaptcha', '1') === '1';
if ($recaptchaEnabled) {
    if (isset($input['recaptcha_token']) && !empty($input['recaptcha_token'])) {
    $recaptchaResult = verifyRecaptcha($input['recaptcha_token']);
    if (!$recaptchaResult['success']) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => $recaptchaResult['message']]);
        logSecurityEvent('recaptcha_verification_failed', 'IP: ' . $clientIP . ', Score: ' . ($recaptchaResult['score'] ?? 'N/A'), 'medium');
        exit();
    }
    
    // Log successful reCAPTCHA verification
    logSecurityEvent('recaptcha_verification_success', 'IP: ' . $clientIP . ', Score: ' . ($recaptchaResult['score'] ?? 'N/A'), 'low');
} else {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'reCAPTCHA token is required']);
    logSecurityEvent('recaptcha_token_missing', 'IP: ' . $clientIP, 'medium');
    exit();
    }
}

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get attachment path if provided (don't sanitize too aggressively - it's a file path)
    // Check if attachment_path exists and is not null, not empty string, and not the string 'null'
    $rawAttachmentPath = $input['attachment_path'] ?? null;
    $attachmentPath = null;
    if ($rawAttachmentPath !== null && $rawAttachmentPath !== '' && $rawAttachmentPath !== 'null' && trim($rawAttachmentPath) !== '') {
        $attachmentPath = trim($rawAttachmentPath);
        // Only sanitize potentially dangerous characters, but keep the path structure
        $attachmentPath = preg_replace('/[<>"\']/', '', $attachmentPath); // Remove dangerous chars but keep slashes and dots
    }
    
    // Check if attachment_path column exists in the table
    $table = $formType === 'bug_report' ? 'bug_reports' : 'feature_requests';
    try {
        $checkColumnQuery = "SHOW COLUMNS FROM $table LIKE 'attachment_path'";
        $checkColumnStmt = $conn->query($checkColumnQuery);
        $hasAttachmentColumn = $checkColumnStmt->rowCount() > 0;
        $checkColumnStmt->closeCursor();
    } catch (Exception $e) {
        $hasAttachmentColumn = false;
    }
    
    // Only use attachment_path if column exists and path is provided
    $useAttachment = $hasAttachmentColumn && !empty($attachmentPath);
    
    // Insert into database
    if ($formType === 'bug_report') {
        if ($useAttachment) {
            $insertQuery = "INSERT INTO bug_reports (name, email, subject, description, priority, status, ip_address, user_agent, attachment_path) 
                           VALUES (?, ?, ?, ?, ?, 'new', ?, ?, ?)";
        } else {
            $insertQuery = "INSERT INTO bug_reports (name, email, subject, description, priority, status, ip_address, user_agent) 
                           VALUES (?, ?, ?, ?, ?, 'new', ?, ?)";
        }
    } else {
        if ($useAttachment) {
            $insertQuery = "INSERT INTO feature_requests (name, email, feature, description, priority, status, ip_address, user_agent, attachment_path) 
                           VALUES (?, ?, ?, ?, ?, 'new', ?, ?, ?)";
        } else {
            $insertQuery = "INSERT INTO feature_requests (name, email, feature, description, priority, status, ip_address, user_agent) 
                           VALUES (?, ?, ?, ?, ?, 'new', ?, ?)";
        }
    }
    
    $stmt = $conn->prepare($insertQuery);
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    
    try {
        if ($useAttachment) {
            $result = $stmt->execute([$name, $email, $subject, $description, $priority, $clientIP, $userAgent, $attachmentPath]);
        } else {
            $result = $stmt->execute([$name, $email, $subject, $description, $priority, $clientIP, $userAgent]);
        }
    } catch (PDOException $e) {
        error_log("Database insert error: " . $e->getMessage());
        throw new Exception("خطا در ذخیره اطلاعات در دیتابیس: " . $e->getMessage());
    }
    
    $submissionId = $conn->lastInsertId();
    
    // Verify the saved data - ALWAYS check, even if useAttachment was false
    $savedAttachmentPath = null;
    if ($hasAttachmentColumn && !empty($attachmentPath)) {
        try {
            // First verify what was actually saved
            $verifyQuery = "SELECT attachment_path FROM " . ($formType === 'bug_report' ? 'bug_reports' : 'feature_requests') . " WHERE id = ?";
            $verifyStmt = $conn->prepare($verifyQuery);
            $verifyStmt->execute([$submissionId]);
            $savedData = $verifyStmt->fetch(PDO::FETCH_ASSOC);
            $verifyStmt->closeCursor();
            $savedAttachmentPath = $savedData ? ($savedData['attachment_path'] ?? null) : null;
            
            // If attachment_path was provided but not saved, try to update it
            if (!empty($attachmentPath) && ($savedAttachmentPath === null || trim($savedAttachmentPath) === '')) {
                try {
                    $updateQuery = "UPDATE " . ($formType === 'bug_report' ? 'bug_reports' : 'feature_requests') . " SET attachment_path = ? WHERE id = ?";
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->execute([$attachmentPath, $submissionId]);
                    $updateStmt->closeCursor();
                    $savedAttachmentPath = $attachmentPath;
                } catch (PDOException $e) {
                    // Silently continue if update fails
                }
            }
        } catch (Exception $e) {
            // Continue execution even if verification fails
        }
    }
    
    // Use the verified saved path for emails
    $savedAttachmentPath = $savedAttachmentPath ?? $attachmentPath;
    
    // Send email notification using improved function
    $emailData = [
        'name' => $name,
        'email' => $email,
        'subject' => $subject,
        'priority' => $priority,
        'description' => $description,
        'ip' => $clientIP,
        'attachment_path' => !empty($savedAttachmentPath) ? $savedAttachmentPath : ($useAttachment ? $attachmentPath : null)
    ];
    
    if ($formType === 'feature_request') {
        $emailData['feature'] = $subject;
    }
    
    // Get admin email from settings
    $settingsQuery = "SELECT setting_value FROM system_settings WHERE setting_key = 'admin_email'";
    $settingsStmt = $conn->query($settingsQuery);
    $adminEmail = $settingsStmt->fetchColumn() ?: 'info@gavahito.ir';
    
    $emailSent = sendNotificationEmail($formType, $emailData);
    
    // Log the submission
    $logMessage = $formType === 'bug_report' ? 'Bug report submitted' : 'Feature request submitted';
    logSecurityEvent('form_submission_successful', $logMessage . ' - ID: ' . $submissionId . ', IP: ' . $clientIP, 'low');
    
    // Send confirmation email to user
    $userSubject = $formType === 'bug_report' ? '✅ تایید گزارش مشکل' : '✅ تایید درخواست ویژگی';
    $userBody = '
    <h3 style="color: #041c3f; margin-bottom: 20px;">✅ ' . ($formType === 'bug_report' ? 'گزارش مشکل' : 'درخواست ویژگی') . ' شما با موفقیت دریافت شد</h3>
    
    <div class="info-row">
        <span class="info-label">👤 نام:</span>
        <span class="info-value">' . htmlspecialchars($name, ENT_QUOTES, 'UTF-8') . '</span>
    </div>
    
    <div class="info-row">
        <span class="info-label">📝 موضوع:</span>
        <span class="info-value">' . htmlspecialchars($subject, ENT_QUOTES, 'UTF-8') . '</span>
    </div>
    
    <div class="info-row">
        <span class="info-label">⚡ اولویت:</span>
        <span class="info-value">
            <span class="priority-badge" style="background-color: ' . getPriorityColor($priority) . '">' . getPriorityLabel($priority) . '</span>
        </span>
    </div>
    
    <div class="info-row">
        <span class="info-label">🔢 شماره پیگیری:</span>
        <span class="info-value">' . $submissionId . '</span>
    </div>
    
    <div style="margin-top: 20px; padding: 15px; background: #e8f5e8; border: 1px solid #4caf50; border-radius: 8px;">
        <p style="margin: 0; color: #2e7d32; font-weight: 600;">📞 ما به زودی با شما تماس خواهیم گرفت</p>
    </div>
    
    <div style="margin-top: 20px; padding: 15px; background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px;">
        <p style="margin: 0; color: #856404;">💡 برای پیگیری بیشتر می‌توانید از شماره پیگیری بالا استفاده کنید</p>
    </div>';
    
    $userEmailSent = sendEmail($email, $userSubject, $userBody);
    
    // Clean output buffer before sending response
    ob_clean();
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => $formType === 'bug_report' ? 'گزارش مشکل با موفقیت ارسال شد' : 'درخواست ویژگی با موفقیت ارسال شد',
        'submission_id' => $submissionId,
        'email_sent' => $emailSent,
        'user_email_sent' => $userEmailSent
    ], JSON_UNESCAPED_UNICODE);
    
    ob_end_flush();
    
} catch (PDOException $e) {
    ob_clean();
    error_log("Form submission PDO error: " . $e->getMessage() . " | Code: " . $e->getCode() . " | SQL State: " . $e->getCode());
    try {
        logSecurityEvent('form_submission_error', 'PDO Error: ' . $e->getMessage() . ', IP: ' . ($clientIP ?? 'unknown'), 'high');
    } catch (Exception $logError) {
        error_log("Failed to log security event: " . $logError->getMessage());
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'خطا در پردازش درخواست. لطفاً دوباره تلاش کنید.',
        'error' => $e->getMessage(),
        'code' => $e->getCode()
    ], JSON_UNESCAPED_UNICODE);
    ob_end_flush();
} catch (Exception $e) {
    ob_clean();
    error_log("Form submission error: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine());
    try {
        logSecurityEvent('form_submission_error', 'Error: ' . $e->getMessage() . ', IP: ' . ($clientIP ?? 'unknown'), 'high');
    } catch (Exception $logError) {
        error_log("Failed to log security event: " . $logError->getMessage());
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'خطا در پردازش درخواست. لطفاً دوباره تلاش کنید.',
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    ob_end_flush();
} catch (Error $e) {
    ob_clean();
    error_log("Form submission fatal error: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'خطا در پردازش درخواست. لطفاً دوباره تلاش کنید.',
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    ob_end_flush();
}
?>
