<?php
/**
 * Secure File Upload API
 * API امن برای آپلود فایل (فقط PNG و JPG)
 */

// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Set error handler to catch all errors
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    error_log("PHP Error [$errno]: $errstr in $errfile on line $errline");
    return false; // Let PHP handle it normally
});

// Set exception handler
set_exception_handler(function($exception) {
    error_log("Uncaught exception: " . $exception->getMessage() . " in " . $exception->getFile() . " on line " . $exception->getLine());
});

// Start output buffering to catch any unexpected output
ob_start();

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

try {
    // Include required files with error handling
    $dbPath = __DIR__ . '/../config/database.php';
    if (!file_exists($dbPath)) {
        throw new Exception("Database config file not found at: $dbPath");
    }
    require_once $dbPath;
    
    $funcPath = __DIR__ . '/../includes/functions.php';
    if (!file_exists($funcPath)) {
        throw new Exception("Functions file not found at: $funcPath");
    }
    require_once $funcPath;
    
    $recaptchaPath = __DIR__ . '/../config/recaptcha-config.php';
    if (!file_exists($recaptchaPath)) {
        throw new Exception("reCAPTCHA config file not found at: $recaptchaPath");
    }
    require_once $recaptchaPath;
} catch (Exception $e) {
    ob_clean();
    $errorMsg = "Upload API include error: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine();
    error_log($errorMsg);
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'خطا در بارگذاری فایل‌های سیستم',
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    exit();
} catch (Error $e) {
    ob_clean();
    $errorMsg = "Upload API fatal error: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine();
    error_log($errorMsg);
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'خطا در بارگذاری فایل‌های سیستم',
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    ob_clean();
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Wrap entire upload logic in try-catch
try {
    // Security checks
    try {
        if (!function_exists('getClientIP')) {
            ob_clean();
            error_log("getClientIP function not found. Functions file may not be loaded correctly.");
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'خطا در بارگذاری توابع سیستم',
                'error' => 'getClientIP function not found'
            ]);
            exit();
        }
        $clientIP = getClientIP();
        if (empty($clientIP)) {
            $clientIP = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        }
    } catch (Exception $e) {
        ob_clean();
        error_log("getClientIP error: " . $e->getMessage() . " | Trace: " . $e->getTraceAsString());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'خطا در دریافت اطلاعات IP',
            'error' => $e->getMessage()
        ]);
        exit();
    } catch (Error $e) {
        ob_clean();
        error_log("getClientIP fatal error: " . $e->getMessage() . " | Trace: " . $e->getTraceAsString());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'خطا در دریافت اطلاعات IP',
            'error' => $e->getMessage()
        ]);
        exit();
    }

    // Check if IP is blacklisted
    try {
        if (isIPBlacklisted($clientIP)) {
            ob_clean();
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            try {
                logSecurityEvent('file_upload_blocked_blacklisted_ip', 'IP: ' . $clientIP, 'high');
            } catch (Exception $e) {
                error_log("Log security event error: " . $e->getMessage());
            }
            exit();
        }
    } catch (Exception $e) {
        error_log("isIPBlacklisted error: " . $e->getMessage());
        // Continue if check fails - don't block upload
    }

    // Rate limiting (increased limit for file uploads: 15 per 5 minutes)
    try {
        if (!checkRateLimit('file_upload', 15)) {
            ob_clean();
            http_response_code(429);
            echo json_encode([
                'success' => false, 
                'message' => 'تعداد درخواست‌های آپلود بیش از حد مجاز است. لطفاً چند دقیقه صبر کنید و دوباره تلاش کنید.',
                'retry_after' => 300 // 5 minutes in seconds
            ]);
            try {
                logSecurityEvent('file_upload_rate_limit_exceeded', 'IP: ' . $clientIP, 'medium');
            } catch (Exception $e) {
                error_log("Log security event error: " . $e->getMessage());
            }
            exit();
        }
    } catch (Exception $e) {
        error_log("checkRateLimit error: " . $e->getMessage());
        // Continue if check fails - don't block upload
    }

    // Verify reCAPTCHA if enabled (for FormData uploads, token is in POST)
    try {
        $recaptchaEnabled = getSystemSetting('enable_recaptcha', '1') === '1';
        if ($recaptchaEnabled) {
            $recaptchaToken = $_POST['recaptcha_token'] ?? '';
            
            if (empty($recaptchaToken)) {
                // Log but allow upload to proceed (reCAPTCHA might be optional)
                error_log("reCAPTCHA token missing for upload from IP: " . $clientIP);
                // Continue with upload - don't block if token is missing
            } else {
                $recaptchaResult = verifyRecaptcha($recaptchaToken);
                if (!$recaptchaResult['success']) {
                    ob_clean();
                    http_response_code(400);
                    echo json_encode(['success' => false, 'message' => $recaptchaResult['message']]);
                    try {
                        logSecurityEvent('recaptcha_verification_failed_upload', 'IP: ' . $clientIP . ', Score: ' . ($recaptchaResult['score'] ?? 'N/A'), 'medium');
                    } catch (Exception $e) {
                        error_log("Log security event error: " . $e->getMessage());
                    }
                    exit();
                }
                try {
                    logSecurityEvent('recaptcha_verification_success_upload', 'IP: ' . $clientIP . ', Score: ' . ($recaptchaResult['score'] ?? 'N/A'), 'low');
                } catch (Exception $e) {
                    error_log("Log security event error: " . $e->getMessage());
                }
            }
        }
    } catch (Exception $e) {
        error_log("reCAPTCHA check error: " . $e->getMessage());
        // Continue if reCAPTCHA check fails - don't block upload
    }

    // Check if file was uploaded
    if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
        $uploadError = $_FILES['file']['error'] ?? 'UNKNOWN';
        $errorMessages = [
            UPLOAD_ERR_INI_SIZE => 'فایل از حداکثر حجم مجاز بزرگ‌تر است',
            UPLOAD_ERR_FORM_SIZE => 'فایل از حداکثر حجم مجاز فرم بزرگ‌تر است',
            UPLOAD_ERR_PARTIAL => 'فایل به صورت ناقص آپلود شده است',
            UPLOAD_ERR_NO_FILE => 'هیچ فایلی انتخاب نشده است',
            UPLOAD_ERR_NO_TMP_DIR => 'پوشه موقت وجود ندارد',
            UPLOAD_ERR_CANT_WRITE => 'نمی‌توان فایل را نوشت',
            UPLOAD_ERR_EXTENSION => 'آپلود فایل توسط extension متوقف شد'
        ];
        $errorMsg = $errorMessages[$uploadError] ?? 'فایل آپلود نشده است';
        ob_clean();
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => $errorMsg]);
        exit();
    }

    $file = $_FILES['file'];

    // File validation
    $allowedTypes = ['image/png', 'image/jpeg', 'image/jpg'];
    $allowedExtensions = ['png', 'jpg', 'jpeg'];
    $maxFileSize = 5 * 1024 * 1024; // 5MB

    // Check file size
    if ($file['size'] > $maxFileSize) {
        ob_clean();
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'حجم فایل نباید بیشتر از 5 مگابایت باشد']);
        exit();
    }

    // Check file extension first (needed for MIME type fallback)
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($extension, $allowedExtensions)) {
        ob_clean();
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'فقط فایل‌های PNG و JPG مجاز هستند']);
        try {
            logSecurityEvent('invalid_file_extension_upload', 'IP: ' . $clientIP . ', Extension: ' . $extension, 'medium');
        } catch (Exception $e) {
            error_log("Log security event error: " . $e->getMessage());
        }
        exit();
    }

    // Check MIME type (with multiple fallbacks)
    $mimeType = null;
    try {
        // Try finfo_open first (most reliable)
        if (function_exists('finfo_open')) {
            $finfo = @finfo_open(FILEINFO_MIME_TYPE);
            if ($finfo !== false) {
                $mimeType = @finfo_file($finfo, $file['tmp_name']);
                @finfo_close($finfo);
            }
        }
        
        // Fallback to mime_content_type if finfo failed or not available
        if (empty($mimeType) && function_exists('mime_content_type')) {
            $mimeType = @mime_content_type($file['tmp_name']);
        }
        
        // Last resort: use file extension (we'll verify with getimagesize later)
        if (empty($mimeType)) {
            $mimeType = $extension === 'png' ? 'image/png' : ($extension === 'jpg' || $extension === 'jpeg' ? 'image/jpeg' : '');
        }
    } catch (Exception $e) {
        error_log("MIME type check error: " . $e->getMessage());
        // Fallback to extension-based check
        $mimeType = $extension === 'png' ? 'image/png' : ($extension === 'jpg' || $extension === 'jpeg' ? 'image/jpeg' : '');
    } catch (Error $e) {
        error_log("MIME type check fatal error: " . $e->getMessage());
        // Fallback to extension-based check
        $mimeType = $extension === 'png' ? 'image/png' : ($extension === 'jpg' || $extension === 'jpeg' ? 'image/jpeg' : '');
    }

    // Verify it's actually an image using getimagesize (most reliable method)
    // This will also give us the correct MIME type
    // We don't check MIME type before this because finfo might not be available
    try {
        $imageInfo = @getimagesize($file['tmp_name']);
        if ($imageInfo === false) {
            ob_clean();
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'فایل انتخاب شده یک تصویر معتبر نیست']);
            try {
                logSecurityEvent('invalid_image_file_upload', 'IP: ' . $clientIP, 'medium');
            } catch (Exception $e) {
                error_log("Log security event error: " . $e->getMessage());
            }
            exit();
        }
        
        // Update MIME type from getimagesize if available (more reliable than finfo/mime_content_type)
        if (isset($imageInfo['mime']) && !empty($imageInfo['mime'])) {
            $detectedMimeType = $imageInfo['mime'];
            if (in_array($detectedMimeType, $allowedTypes)) {
                $mimeType = $detectedMimeType;
            } else {
                // Image is valid but wrong type
                ob_clean();
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'فقط فایل‌های PNG و JPG مجاز هستند']);
                try {
                    logSecurityEvent('invalid_image_type_upload', 'IP: ' . $clientIP . ', MIME: ' . $detectedMimeType, 'medium');
                } catch (Exception $e) {
                    error_log("Log security event error: " . $e->getMessage());
                }
                exit();
            }
        }
    } catch (Exception $e) {
        ob_clean();
        error_log("getimagesize error: " . $e->getMessage());
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'خطا در بررسی تصویر']);
        exit();
    } catch (Error $e) {
        ob_clean();
        error_log("getimagesize fatal error: " . $e->getMessage());
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'خطا در بررسی تصویر']);
        exit();
    }

    // Check image dimensions (prevent extremely large images)
    $maxWidth = 4000;
    $maxHeight = 4000;
    if ($imageInfo[0] > $maxWidth || $imageInfo[1] > $maxHeight) {
        ob_clean();
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'ابعاد تصویر نباید بیشتر از 4000x4000 پیکسل باشد']);
        exit();
    }

    // Generate secure filename
    $secureFilename = bin2hex(random_bytes(16)) . '_' . time() . '.' . $extension;

    // Create upload directory if it doesn't exist
    $uploadDir = __DIR__ . '/../uploads/bug-reports/';
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            ob_clean();
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'خطا در ایجاد پوشه آپلود. لطفاً مجوزهای پوشه را بررسی کنید.']);
            try {
                logSecurityEvent('file_upload_directory_creation_failed', 'IP: ' . $clientIP . ', Path: ' . $uploadDir, 'high');
            } catch (Exception $e) {
                error_log("Log security event error: " . $e->getMessage());
            }
            exit();
        }
    }

    // Check if directory is writable
    if (!is_writable($uploadDir)) {
        ob_clean();
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'پوشه آپلود قابل نوشتن نیست. لطفاً مجوزهای پوشه را بررسی کنید.']);
        try {
            logSecurityEvent('file_upload_directory_not_writable', 'IP: ' . $clientIP . ', Path: ' . $uploadDir, 'high');
        } catch (Exception $e) {
            error_log("Log security event error: " . $e->getMessage());
        }
        exit();
    }

    // .htaccess file is already created in the uploads directory
    // فایل .htaccess قبلاً در پوشه uploads ایجاد شده است

    $uploadPath = $uploadDir . $secureFilename;

    // Move uploaded file
    try {
        if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
            $error = error_get_last();
            $errorMsg = $error['message'] ?? 'Unknown error';
            error_log("File upload failed: " . $errorMsg . " - Source: " . $file['tmp_name'] . " - Dest: " . $uploadPath);
            error_log("Upload directory exists: " . (is_dir($uploadDir) ? 'yes' : 'no'));
            error_log("Upload directory writable: " . (is_writable($uploadDir) ? 'yes' : 'no'));
            if (is_dir($uploadDir)) {
                error_log("Upload directory permissions: " . substr(sprintf('%o', fileperms($uploadDir)), -4));
            }
            ob_clean();
            http_response_code(500);
            echo json_encode([
                'success' => false, 
                'message' => 'خطا در آپلود فایل. لطفاً مجوزهای پوشه را بررسی کنید.',
                'error' => $errorMsg
            ]);
            try {
                logSecurityEvent('file_upload_failed', 'IP: ' . $clientIP . ' - Error: ' . $errorMsg, 'high');
            } catch (Exception $e) {
                error_log("Log security event error: " . $e->getMessage());
            }
            exit();
        }
    } catch (Exception $e) {
        ob_clean();
        error_log("move_uploaded_file exception: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false, 
            'message' => 'خطا در آپلود فایل: ' . $e->getMessage()
        ]);
        exit();
    }

    // Verify file was saved correctly
    if (!file_exists($uploadPath) || filesize($uploadPath) === 0) {
        ob_clean();
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'خطا در ذخیره فایل']);
        exit();
    }

    // Log successful upload
    try {
        logSecurityEvent('file_upload_successful', 'IP: ' . $clientIP . ', File: ' . $secureFilename, 'low');
    } catch (Exception $e) {
        error_log("Log security event error: " . $e->getMessage());
        // Don't fail the upload if logging fails
    }

    // Clear output buffer before sending response
    ob_clean();

    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'فایل با موفقیت آپلود شد',
        'filename' => $secureFilename,
        'path' => 'uploads/bug-reports/' . $secureFilename,
        'size' => $file['size'],
        'mime_type' => $mimeType
    ]);
    exit();

} catch (Exception $e) {
    // Catch any unexpected errors
    ob_clean();
    error_log("Upload API unexpected error: " . $e->getMessage() . " | Trace: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'خطا در آپلود فایل. لطفاً دوباره تلاش کنید.',
        'error' => $e->getMessage()
    ]);
    exit();
} catch (Error $e) {
    // Catch PHP fatal errors
    ob_clean();
    error_log("Upload API fatal error: " . $e->getMessage() . " | Trace: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'خطا در آپلود فایل. لطفاً دوباره تلاش کنید.',
        'error' => $e->getMessage()
    ]);
    exit();
}
?>

