<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to backups.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

// Check for error messages from URL
if (isset($_GET['error'])) {
    $error = $_GET['error'];
}

// Handle backup actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        switch ($action) {
            case 'create_backup':
                $backupType = $_POST['backup_type'] ?? 'full';
                $result = createBackup($backupType);
                
                if ($result['success']) {
                    $message = "بکاپ با موفقیت ایجاد شد: " . $result['backup_name'];
                } else {
                    $error = "خطا در ایجاد بکاپ: " . $result['error'];
                }
                break;
                
            case 'delete_backup':
                $backupName = $_POST['backup_name'] ?? '';
                if ($backupName && deleteBackup($backupName)) {
                    $message = "بکاپ با موفقیت حذف شد";
                } else {
                    $error = "خطا در حذف بکاپ";
                }
                break;
                
            case 'delete_multiple_backups':
                $backupNames = $_POST['backup_names'] ?? [];
                if (!empty($backupNames)) {
                    $successCount = 0;
                    $errorCount = 0;
                    
                    foreach ($backupNames as $backupName) {
                        if (deleteBackup($backupName)) {
                            $successCount++;
                        } else {
                            $errorCount++;
                        }
                    }
                    
                    if ($errorCount === 0) {
                        $message = "{$successCount} بکاپ با موفقیت حذف شد";
                    } else {
                        $message = "{$successCount} بکاپ حذف شد، {$errorCount} بکاپ با خطا مواجه شد";
                    }
                } else {
                    $error = "هیچ بکاپی انتخاب نشده است";
                }
                break;
                

                
            case 'clean_old_auto_backups':
                // Manually clean old automatic backups
                $backups = getBackups();
                $automaticBackups = array_filter($backups, function($backup) {
                    return preg_match('/_auto$/', $backup['backup_name']);
                });
                
                if (count($automaticBackups) > 5) {
                    // Sort by creation date (newest first)
                    usort($automaticBackups, function($a, $b) {
                        return strtotime($b['created_at']) - strtotime($a['created_at']);
                    });
                    
                    // Get backups to delete (keep the first 5)
                    $backupsToDelete = array_slice($automaticBackups, 5);
                    
                    $deletedCount = 0;
                    foreach ($backupsToDelete as $backup) {
                        if (deleteBackup($backup['backup_name'])) {
                            $deletedCount++;
                        }
                    }
                    
                    $message = "{$deletedCount} بکاپ خودکار قدیمی حذف شد. 5 بکاپ اخیر نگه داشته شد.";
                } else {
                    $message = "هیچ بکاپ خودکار قدیمی برای حذف وجود ندارد.";
                }
                break;
                

                

        }
    } catch (Exception $e) {
        error_log("Backup action error: " . $e->getMessage());
        $error = 'خطا در عملیات. لطفاً دوباره تلاش کنید';
    }
}

// Get backup statistics
$backups = [];
$totalBackups = 0;
$totalSize = 0;
$lastBackup = null;
$shouldRunAutoBackup = false;

// Get automatic backup settings
$backupEnabled = getSystemSetting('backup_enabled');
$backupFrequency = getSystemSetting('backup_frequency');
$lastBackupTime = getLastBackupTime();

try {
    $backups = getBackups();
    $totalBackups = count($backups);
    
    foreach ($backups as $backup) {
        $totalSize += $backup['total_size'];
        if (!$lastBackup || strtotime($backup['created_at']) > strtotime($lastBackup['created_at'])) {
            $lastBackup = $backup;
        }
    }
    
    $shouldRunAutoBackup = shouldRunAutomaticBackup();
} catch (Exception $e) {
    $error = "خطا در بارگذاری اطلاعات بکاپ: " . $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت بکاپ‌ها - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    <?php include 'includes/admin-styles.php'; ?>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">📦 مدیریت بکاپ‌ها</h1>
            <p class="page-description">مدیریت و مشاهده بکاپ‌های سیستم</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="card" style="margin-bottom: 2rem;">
            <div class="card-header">
                <h2 class="card-title">📊 آمار بکاپ‌ها</h2>

            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1.5rem;">
                <div style="display: flex; align-items: center; gap: 1rem; padding: 1.5rem;  border-radius: 12px; border: 1px solid var(--gray-200);">
                    <div style="font-size: 2.5rem; opacity: 0.8;">📊</div>
                    <div style="flex: 1;">
                        <div style="font-size: 1.5rem; font-weight: bold; color: var(--dark-300); margin-bottom: 0.25rem;"><?php echo $totalBackups; ?></div>
                        <div style="font-size: 0.875rem; color: var(--dark-400);">کل بکاپ‌ها</div>
                    </div>
                </div>

                <div style="display: flex; align-items: center; gap: 1rem; padding: 1.5rem; background: linear-gradient(135deg, var(--success-50) 0%, var(--success-100) 100%); border-radius: 12px; border: 1px solid var(--success-200);">
                    <div style="font-size: 2.5rem; opacity: 0.8;">💾</div>
                    <div style="flex: 1;">
                        <div style="font-size: 1.5rem; font-weight: bold; color: var(--dark-300); margin-bottom: 0.25rem;"><?php echo formatFileSize($totalSize); ?></div>
                        <div style="font-size: 0.875rem; color: var(--dark-400);">حجم کل</div>
                    </div>
                </div>

                <div style="display: flex; align-items: center; gap: 1rem; padding: 1.5rem; background: linear-gradient(135deg, var(--warning-50) 0%, var(--warning-100) 100%); border-radius: 12px; border: 1px solid var(--warning-200);">
                    <div style="font-size: 2.5rem; opacity: 0.8;">🕒</div>
                    <div style="flex: 1;">
                        <div style="font-size: 1.5rem; font-weight: bold; color: var(--dark-300); margin-bottom: 0.25rem;">
                            <?php echo $lastBackup ? formatDate($lastBackup['created_at']) : 'هیچ بکاپی'; ?>
                        </div>
                        <div style="font-size: 0.875rem; color: var(--dark-400);">آخرین بکاپ</div>
                    </div>
                </div>

                <div style="display: flex; align-items: center; gap: 1rem; padding: 1.5rem; background: linear-gradient(135deg, var(--info-50) 0%, var(--info-100) 100%); border-radius: 12px; border: 1px solid var(--info-200);">
                    <div style="font-size: 2.5rem; opacity: 0.8;">⚡</div>
                    <div style="flex: 1;">
                        <div style="font-size: 1.5rem; font-weight: bold; color: var(--dark-300); margin-bottom: 0.25rem; <?php echo $shouldRunAutoBackup ? 'color: var(--warning-600) !important;' : 'color: var(--success-600) !important;'; ?>">
                            <?php echo $shouldRunAutoBackup ? 'نیاز به بکاپ' : 'به‌روز'; ?>
                        </div>
                        <div style="font-size: 0.875rem; color: var(--dark-400);">وضعیت خودکار</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Automatic Backup Settings -->
        <div class="card" style="margin-bottom: 2rem;">
            <div class="card-header">
                <h2 class="card-title">⚙️ تنظیمات پشتیبان‌گیری خودکار</h2>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1.5rem;">
                <div style="padding: 1.5rem; background: <?php echo $backupEnabled === '1' ? 'linear-gradient(135deg, var(--success-50) 0%, var(--success-100) 100%)' : 'linear-gradient(135deg, var(--gray-50) 0%, var(--gray-100) 100%)'; ?>; border-radius: 12px; border: 1px solid <?php echo $backupEnabled === '1' ? 'var(--success-200)' : 'var(--gray-200)'; ?>;">
                    <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                        <div style="font-size: 2rem; opacity: 0.8;"><?php echo $backupEnabled === '1' ? '✅' : '❌'; ?></div>
                        <div>
                            <div style="font-weight: bold; color: var(--dark-300); margin-bottom: 0.25rem;">وضعیت پشتیبان‌گیری خودکار</div>
                            <div style="font-size: 0.875rem; color: var(--dark-400);"><?php echo $backupEnabled === '1' ? 'فعال' : 'غیرفعال'; ?></div>
                        </div>
                    </div>
                    <a href="settings.php#backup-settings" class="btn btn-sm <?php echo $backupEnabled === '1' ? 'btn-success' : 'btn-secondary'; ?>" style="width: 100%;">
                        <?php echo $backupEnabled === '1' ? 'تغییر تنظیمات' : 'فعال‌سازی'; ?>
                    </a>
                </div>

                <div style="padding: 1.5rem;  border-radius: 12px; border: 1px solid var(--gray-200);">
                    <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                        <div style="font-size: 2rem; opacity: 0.8;">🕒</div>
                        <div>
                            <div style="font-weight: bold; color: var(--dark-300); margin-bottom: 0.25rem;">فرکانس پشتیبان‌گیری</div>
                            <div style="font-size: 0.875rem; color: var(--dark-400);">
                                <?php 
                                switch($backupFrequency) {
                                    case 'daily': echo 'روزانه'; break;
                                    case 'weekly': echo 'هفتگی'; break;
                                    case 'monthly': echo 'ماهانه'; break;
                                    default: echo 'تعریف نشده';
                                }
                                ?>
                            </div>
                        </div>
                    </div>
                    <a href="settings.php#backup-settings" class="btn btn-sm btn-secondary" style="width: 100%;">تغییر فرکانس</a>
                </div>

                <div style="padding: 1.5rem; border-radius: 12px; border: 1px solid var(--gray-200);">
                    <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                        <div style="font-size: 2rem; opacity: 0.8;">📅</div>
                        <div>
                            <div style="font-weight: bold; color: var(--dark-300); margin-bottom: 0.25rem;">آخرین بکاپ خودکار</div>
                            <div style="font-size: 0.875rem; color: var(--dark-400);">
                                <?php echo $lastBackupTime ? formatDate($lastBackupTime) : 'هیچ بکاپی'; ?>
                            </div>
                        </div>
                    </div>
                    <?php if ($backupEnabled === '1' && $shouldRunAutoBackup): ?>
                        <div style="background: var(--warning-50); color: var(--warning-700); padding: 0.5rem; border-radius: 8px; font-size: 0.875rem; text-align: center;">
                            ⚠️ زمان بکاپ خودکار فرا رسیده
                        </div>
                    <?php elseif ($backupEnabled === '1'): ?>
                        <div style="background: var(--success-50); color: var(--success-700); padding: 0.5rem; border-radius: 8px; font-size: 0.875rem; text-align: center;">
                            ✅ بکاپ خودکار به‌روز است
                        </div>
                    <?php else: ?>
                        <div style="background: var(--gray-50); color: var(--gray-700); padding: 0.5rem; border-radius: 8px; font-size: 0.875rem; text-align: center;">
                            ⏸️ پشتیبان‌گیری خودکار غیرفعال است
                        </div>
                    <?php endif; ?>
                </div>

                <?php if ($backupEnabled === '1'): ?>
                <div style="padding: 1.5rem; ">
                    <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                        <div style="font-size: 2rem; opacity: 0.8;">🚀</div>
                        <div>
                            <div style="font-weight: bold; color: var(--dark-300); margin-bottom: 0.25rem;">اجرای دستی بکاپ خودکار</div>
                            <div style="font-size: 0.875rem; color: var(--dark-400);">ایجاد بکاپ کامل طبق تنظیمات</div>
                        </div>
                    </div>
                    <form method="POST" action="" class="backup-form" style="margin: 0;">
                        <input type="hidden" name="action" value="create_backup">
                        <input type="hidden" name="backup_type" value="full">
                        <button type="submit" class="btn btn-primary" style="width: 100%;">
                            🔄 ایجاد بکاپ خودکار
                        </button>
                    </form>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Create Backup Section -->
        <div class="card" style="margin-bottom: 2rem;">
            <div class="card-header">
                <h2 class="card-title">🔄 ایجاد بکاپ جدید</h2>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1rem;">
                <form method="POST" action="" class="backup-form">
                    <input type="hidden" name="action" value="create_backup">
                    <input type="hidden" name="backup_type" value="full">
                    <button type="submit" class="btn btn-primary" style="width: 100%; padding: 1.5rem; text-align: center; border-radius: 12px;">
                        <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">📦</div>
                        <div style="font-weight: bold; margin-bottom: 0.25rem; font-size: 1.1rem;">بکاپ کامل</div>
                        <div style="font-size: 0.9rem; opacity: 0.8;">فایل‌ها + دیتابیس + کل سایت</div>
                    </button>
                </form>

                <form method="POST" action="" class="backup-form">
                    <input type="hidden" name="action" value="create_backup">
                    <input type="hidden" name="backup_type" value="database">
                    <button type="submit" class="btn btn-secondary" style="width: 100%; padding: 1.5rem; text-align: center; border-radius: 12px;">
                        <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">🗄️</div>
                        <div style="font-weight: bold; margin-bottom: 0.25rem; font-size: 1.1rem;">فقط دیتابیس</div>
                        <div style="font-size: 0.9rem; opacity: 0.8;">دیتابیس فقط</div>
                    </button>
                </form>

                <form method="POST" action="" class="backup-form">
                    <input type="hidden" name="action" value="create_backup">
                    <input type="hidden" name="backup_type" value="files">
                    <button type="submit" class="btn btn-secondary" style="width: 100%; padding: 1.5rem; text-align: center; border-radius: 12px;">
                        <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">📁</div>
                        <div style="font-weight: bold; margin-bottom: 0.25rem; font-size: 1.1rem;">فقط فایل‌ها</div>
                        <div style="font-size: 0.9rem; opacity: 0.8;">فایل‌های سایت</div>
                    </button>
                </form>
            </div>
        </div>

        <!-- Advanced Backup Management Panel -->
        <div class="card advanced-panel-card" style="margin-bottom: 2rem; background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(20px); border: 1px solid rgba(255, 255, 255, 0.2);">
            <div class="card-header">
                <h2 class="card-title">🎛️ پنل مدیریت پیشرفته بکاپ</h2>
            </div>
            
            <div class="card-body">
                <!-- Backup Statistics Dashboard -->
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1rem; margin-bottom: 2rem;">
                    <!-- Backup Health Status -->
                    <div class="stat-card" style="background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(15px); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: 12px; padding: 1.5rem; transition: all 0.3s ease;">
                        <div style="display: flex; align-items: center; gap: 1rem;">
                            <div style="font-size: 2.5rem; opacity: 0.8;"><?php echo $backupEnabled === '1' ? '🟢' : '🔴'; ?></div>
                            <div>
                                <div style="font-weight: bold; color: white; margin-bottom: 0.25rem;">وضعیت سیستم</div>
                                <div style="font-size: 0.875rem; color: rgba(255, 255, 255, 0.7);"><?php echo $backupEnabled === '1' ? 'سالم' : 'نیاز به توجه'; ?></div>
                            </div>
                        </div>
                    </div>

                    <!-- Next Backup Time -->
                    <div class="stat-card" style="background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(15px); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: 12px; padding: 1.5rem; transition: all 0.3s ease;">
                        <div style="display: flex; align-items: center; gap: 1rem;">
                            <div style="font-size: 2.5rem; opacity: 0.8;">⏰</div>
                            <div>
                                <div style="font-weight: bold; color: white; margin-bottom: 0.25rem;">بکاپ بعدی</div>
                                <div style="font-size: 0.875rem; color: rgba(255, 255, 255, 0.7);">
                                    <?php 
                                    if ($backupEnabled === '1' && $lastBackupTime) {
                                        $nextBackup = strtotime($lastBackupTime) + 86400; // 24 hours
                                        echo date('Y/m/d H:i', $nextBackup);
                                    } else {
                                        echo 'تنظیم نشده';
                                    }
                                    ?>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Storage Usage -->
                    <div class="stat-card" style="background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(15px); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: 12px; padding: 1.5rem; transition: all 0.3s ease;">
                        <div style="display: flex; align-items: center; gap: 1rem;">
                            <div style="font-size: 2.5rem; opacity: 0.8;">💾</div>
                            <div>
                                <div style="font-weight: bold; color: white; margin-bottom: 0.25rem;">فضای استفاده شده</div>
                                <div style="font-size: 0.875rem; color: rgba(255, 255, 255, 0.7);"><?php echo formatFileSize($totalSize); ?></div>
                            </div>
                        </div>
                    </div>

                    <!-- Auto Clean Status -->
                    <div class="stat-card" style="background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(15px); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: 12px; padding: 1.5rem; transition: all 0.3s ease;">
                        <div style="display: flex; align-items: center; gap: 1rem;">
                            <div style="font-size: 2.5rem; opacity: 0.8;">🧹</div>
                            <div>
                                <div style="font-weight: bold; color: white; margin-bottom: 0.25rem;">پاکسازی خودکار</div>
                                <div style="font-size: 0.875rem; color: rgba(255, 255, 255, 0.7);">5 بکاپ آخر</div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 2rem;">
                    <button onclick="showBackupProgress()" class="btn quick-action-btn" style="padding: 1rem; border-radius: 8px; background: linear-gradient(135deg, var(--primary-500), var(--primary-600)); border: none; color: white;">
                        <div style="font-size: 1.5rem; margin-bottom: 0.5rem;">📊</div>
                        <div style="font-weight: bold;">نمایش پیشرفت</div>
                    </button>
                    
                    <button onclick="testBackupSystem()" class="btn quick-action-btn" style="padding: 1rem; border-radius: 8px; background: linear-gradient(135deg, var(--secondary-500), var(--secondary-600)); border: none; color: white;">
                        <div style="font-size: 1.5rem; margin-bottom: 0.5rem;">🧪</div>
                        <div style="font-weight: bold;">تست سیستم</div>
                    </button>
                    
                    <button onclick="showBackupHistory()" class="btn quick-action-btn" style="padding: 1rem; border-radius: 8px; background: linear-gradient(135deg, var(--accent-500), var(--accent-600)); border: none; color: white;">
                        <div style="font-size: 1.5rem; margin-bottom: 0.5rem;">📈</div>
                        <div style="font-weight: bold;">تاریخچه</div>
                    </button>
                    
                    <button onclick="showBackupSettings()" class="btn quick-action-btn" style="padding: 1rem; border-radius: 8px; background: linear-gradient(135deg, #f59e0b, #d97706); border: none; color: white;">
                        <div style="font-size: 1.5rem; margin-bottom: 0.5rem;">⚙️</div>
                        <div style="font-weight: bold;">تنظیمات</div>
                    </button>
                </div>

                <!-- Backup Progress Modal -->
                <div id="backupProgressModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); backdrop-filter: blur(10px); z-index: 1000;">
                    <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(20px); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: 12px; padding: 2rem; max-width: 500px; width: 90%; color: white;">
                        <h3 style="margin-bottom: 1rem; color: white;">📊 پیشرفت بکاپ</h3>
                        <div id="backupProgressBar" style="width: 100%; height: 20px; background: rgba(255, 255, 255, 0.2); border-radius: 10px; overflow: hidden; margin-bottom: 1rem;">
                            <div id="progressFill" style="width: 0%; height: 100%; background: linear-gradient(90deg, var(--primary-500), var(--success-500)); transition: width 0.3s;"></div>
                        </div>
                        <div id="progressText" style="text-align: center; margin-bottom: 1rem; color: rgba(255, 255, 255, 0.9);">آماده برای شروع...</div>
                        <button onclick="closeModal('backupProgressModal')" class="btn" style="width: 100%; background: linear-gradient(135deg, var(--secondary-500), var(--secondary-600)); color: white; border: none;">بستن</button>
                    </div>
                </div>

                <!-- Backup History Modal -->
                <div id="backupHistoryModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); backdrop-filter: blur(10px); z-index: 1000;">
                    <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(20px); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: 12px; padding: 2rem; max-width: 800px; width: 90%; max-height: 80vh; overflow-y: auto; color: white;">
                        <h3 style="margin-bottom: 1rem; color: white;">📈 تاریخچه بکاپ‌ها</h3>
                        <div id="backupHistoryContent">
                            <!-- Content will be loaded here -->
                        </div>
                        <button onclick="closeModal('backupHistoryModal')" class="btn" style="width: 100%; margin-top: 1rem; background: linear-gradient(135deg, var(--secondary-500), var(--secondary-600)); color: white; border: none;">بستن</button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Automatic Backup Logs -->
        <?php if ($backupEnabled === '1'): ?>
        <div class="card" style="margin-bottom: 2rem;">
            <div class="card-header">
                <h2 class="card-title">📝 لاگ‌های بکاپ خودکار</h2>
            </div>
            
            <div class="card-body">
                <?php
                $logFile = __DIR__ . '/logs/cron-backup.log';
                $logs = [];
                if (file_exists($logFile)) {
                    $logContent = file_get_contents($logFile);
                    $logLines = array_filter(explode("\n", $logContent));
                    $logs = array_slice(array_reverse($logLines), 0, 10); // Last 10 lines
                }
                ?>
                
                <?php if (empty($logs)): ?>
                    <div style="text-align: center; padding: 2rem; color: var(--dark-400);">
                        <div style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.5;">📄</div>
                        <p>هنوز هیچ لاگی ثبت نشده است</p>
                    </div>
                <?php else: ?>
                    <div style="max-height: 300px; overflow-y: auto; background: var(--gray-50); border-radius: 8px; padding: 1rem;">
                        <?php foreach ($logs as $log): ?>
                            <div style="font-family: monospace; font-size: 0.875rem; color: var(--dark-600); margin-bottom: 0.5rem; padding: 0.5rem; background: white; border-radius: 4px; border-left: 3px solid var(--primary-500);">
                                <?php echo htmlspecialchars($log); ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <div style="margin-top: 1rem; text-align: center;">
                        <a href="logs/cron-backup.log" target="_blank" class="btn btn-sm btn-secondary">
                            📄 مشاهده تمام لاگ‌ها
                        </a>

                        <form method="POST" action="" style="display: inline-block; margin-right: 0.5rem;">
                            <input type="hidden" name="action" value="clean_old_auto_backups">
                            <button type="submit" class="btn btn-sm btn-danger">
                                🗑️ پاکسازی بکاپ‌های قدیمی
                            </button>
                        </form>


                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Backups List -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📋 لیست بکاپ‌ها</h2>
            </div>
            
            <div class="card-body">
                <?php if (empty($backups)): ?>
                    <div style="text-align: center; padding: 3rem; color: var(--dark-400);">
                        <div style="font-size: 4rem; margin-bottom: 1rem; opacity: 0.5;">📭</div>
                        <h3 style="font-size: 1.25rem; font-weight: 600; margin-bottom: 0.5rem; color: var(--dark-300);">هیچ بکاپی یافت نشد</h3>
                        <p>هنوز هیچ بکاپی ایجاد نشده است.</p>
                    </div>
                <?php else: ?>
                    <!-- Select All -->
                    <div class="select-all-wrapper">
                        <div class="checkbox-wrapper">
                            <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                            <label for="selectAll" style="font-weight: 500; color: var(--dark-300);">انتخاب همه</label>
                        </div>
                    </div>

                    <!-- Bulk Actions -->
                    <div id="bulkActions" class="bulk-actions">
                        <span style="font-weight: 500; color: var(--warning-700);">
                            <span id="selectedCount">0</span> بکاپ انتخاب شده
                        </span>
                        <button type="button" class="btn btn-sm btn-danger" onclick="deleteSelectedBackups()">
                            🗑️ حذف انتخاب شده‌ها
                        </button>
                    </div>

                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th style="width: 50px;">انتخاب</th>
                                    <th>نام بکاپ</th>
                                    <th>نوع</th>
                                    <th>تاریخ ایجاد</th>
                                    <th>حجم</th>
                                    <th>وضعیت</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($backups as $backup): ?>
                                    <tr>
                                        <td>
                                            <div class="checkbox-wrapper">
                                                <input type="checkbox" class="backup-checkbox" value="<?php echo htmlspecialchars($backup['backup_name']); ?>">
                                            </div>
                                        </td>
                                        <td>
                                            <div style="font-weight: 500; color: var(--dark-300);"><?php echo htmlspecialchars($backup['backup_name']); ?></div>
                                            <div style="font-size: 0.85rem; color: var(--dark-400);"><?php echo $backup['files_count']; ?> فایل</div>
                                        </td>
                                        <td>
                                            <?php
                                            $typeLabels = [
                                                'full' => '<span style="padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 500; background-color: var(--primary-100); color: var(--primary-700);">کامل</span>',
                                                'database' => '<span style="padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 500; background-color: var(--success-100); color: var(--success-700);">دیتابیس</span>',
                                                'files' => '<span style="padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 500; background-color: var(--warning-100); color: var(--warning-700);">فایل‌ها</span>'
                                            ];
                                            echo $typeLabels[$backup['backup_type']] ?? '<span style="padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 500; background-color: var(--gray-100); color: var(--gray-700);">نامشخص</span>';
                                            ?>
                                        </td>
                                        <td style="color: var(--dark-400); font-size: 0.9rem;">
                                            <?php echo formatDate($backup['created_at']); ?>
                                        </td>
                                        <td style="color: var(--dark-400); font-size: 0.9rem;">
                                            <?php echo formatFileSize($backup['total_size']); ?>
                                        </td>
                                        <td>
                                            <?php
                                            $statusLabels = [
                                                'in_progress' => '<span style="padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 500; background-color: var(--warning-100); color: var(--warning-700);">در حال انجام</span>',
                                                'completed' => '<span style="padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 500; background-color: var(--success-100); color: var(--success-700);">تکمیل شده</span>',
                                                'failed' => '<span style="padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 500; background-color: var(--danger-100); color: var(--danger-700);">ناموفق</span>'
                                            ];
                                            echo $statusLabels[$backup['status']] ?? '<span style="padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 500; background-color: var(--gray-100); color: var(--gray-700);">نامشخص</span>';
                                            ?>
                                        </td>
                                        <td>
                                            <div style="display: flex; gap: 0.5rem;">
                                                <a href="download-backup.php?backup=<?php echo urlencode($backup['backup_name']); ?>" 
                                                   class="btn btn-sm btn-primary" 
                                                   title="دانلود"
                                                   style="text-decoration: none;">
                                                    📥
                                                </a>
                                                <form method="POST" action="" style="display: inline;" onsubmit="return confirm('آیا از حذف این بکاپ اطمینان دارید؟')">
                                                    <input type="hidden" name="action" value="delete_backup">
                                                    <input type="hidden" name="backup_name" value="<?php echo htmlspecialchars($backup['backup_name']); ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" title="حذف">
                                                        🗑️
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <?php include 'includes/footer.php'; ?>

    <!-- Progress Modal -->
    <div id="progressModal" class="modal" style="display: none;">
        <div class="modal-content" style="max-width: 500px; text-align: center;">
            <div class="modal-header">
                <h3>🔄 در حال ایجاد بکاپ...</h3>
            </div>
            <div class="modal-body">
                <div style="margin-bottom: 1rem;">
                    <div style="font-size: 2rem; margin-bottom: 0.5rem;">📦</div>
                    <div id="backupStatus">آماده‌سازی...</div>
                </div>
                
                <div style="background: var(--gray-100); border-radius: 8px; overflow: hidden; margin-bottom: 1rem;">
                    <div id="progressBar" style="height: 20px; background: linear-gradient(90deg, var(--primary-500), var(--primary-600)); width: 0%; transition: width 0.3s ease;"></div>
                </div>
                
                <div style="display: flex; justify-content: space-between; font-size: 0.875rem; color: var(--dark-400);">
                    <span id="progressText">0%</span>
                    <span id="progressDetails">آماده‌سازی...</span>
                </div>
            </div>
        </div>
    </div>

    <style>
    .modal {
        position: fixed;
        z-index: 9999;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
    }
    
    .modal-content {
        background: white;
        border-radius: 12px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
        overflow: hidden;
    }
    
    .modal-header {
        padding: 1.5rem;
        border-bottom: 1px solid var(--gray-200);
        background: linear-gradient(135deg, var(--primary-50) 0%, var(--primary-100) 100%);
    }
    
    .modal-header h3 {
        margin: 0;
        color: var(--primary-700);
    }
    
    .modal-body {
        padding: 2rem;
    }
    
    .checkbox-wrapper {
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }
    
    .checkbox-wrapper input[type="checkbox"] {
        width: 18px;
        height: 18px;
        accent-color: var(--primary-600);
    }
    
    .bulk-actions {
        display: none;
        gap: 0.5rem;
        margin-bottom: 1rem;
        padding: 1rem;
        background: var(--warning-50);
        border: 1px solid var(--warning-200);
        border-radius: 8px;
    }
    
    .bulk-actions.show {
        display: flex;
        align-items: center;
    }
    
    .select-all-wrapper {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        margin-bottom: 1rem;
        padding: 0.75rem;
        background: var(--gray-50);
        border-radius: 8px;
    }
    </style>

    <script>
    let progressInterval;
    let currentProgress = 0;
    let isBackupRunning = false;

    // Show progress modal
    function showProgressModal() {
        document.getElementById('progressModal').style.display = 'flex';
        currentProgress = 0;
        updateProgress(0, 'آماده‌سازی...');
        startProgressSimulation();
    }

    // Update progress
    function updateProgress(percent, status) {
        document.getElementById('progressBar').style.width = percent + '%';
        document.getElementById('progressText').textContent = percent + '%';
        document.getElementById('backupStatus').textContent = status;
        document.getElementById('progressDetails').textContent = status;
        currentProgress = percent;
    }

    // Simulate progress
    function startProgressSimulation() {
        let steps = [
            { progress: 10, status: 'بررسی فایل‌ها...' },
            { progress: 25, status: 'کپی فایل‌های سایت...' },
            { progress: 50, status: 'بکاپ دیتابیس...' },
            { progress: 75, status: 'فشرده‌سازی...' },
            { progress: 90, status: 'ذخیره اطلاعات...' },
            { progress: 100, status: 'تکمیل شد!' }
        ];
        
        let currentStep = 0;
        
        progressInterval = setInterval(() => {
            if (currentStep < steps.length) {
                updateProgress(steps[currentStep].progress, steps[currentStep].status);
                currentStep++;
            } else {
                clearInterval(progressInterval);
                setTimeout(() => {
                    hideProgressModal();
                    location.reload();
                }, 1000);
            }
        }, 800);
    }

    // Hide progress modal
    function hideProgressModal() {
        document.getElementById('progressModal').style.display = 'none';
        if (progressInterval) {
            clearInterval(progressInterval);
        }
    }

    // Handle backup form submission
    function handleBackupSubmit(form) {
        if (isBackupRunning) return false;
        
        isBackupRunning = true;
        showProgressModal();
        
        // Submit form after a short delay to show progress
        setTimeout(() => {
            form.submit();
        }, 500);
        
        return false;
    }

    // Advanced Backup Management Functions
    function showBackupProgress() {
        document.getElementById('backupProgressModal').style.display = 'block';
        startProgressSimulation();
    }

    function testBackupSystem() {
        if (confirm('آیا می‌خواهید سیستم بکاپ را تست کنید؟')) {
            // Create test backup
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '';
            
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'create_backup';
            form.appendChild(actionInput);
            
            const typeInput = document.createElement('input');
            typeInput.type = 'hidden';
            typeInput.name = 'backup_type';
            typeInput.value = 'test';
            form.appendChild(typeInput);
            
            document.body.appendChild(form);
            form.submit();
        }
    }

    function showBackupHistory() {
        document.getElementById('backupHistoryModal').style.display = 'block';
        loadBackupHistory();
    }

    function showBackupSettings() {
        window.location.href = 'settings.php#backup-settings';
    }

    function closeModal(modalId) {
        document.getElementById(modalId).style.display = 'none';
    }

    function loadBackupHistory() {
        const content = document.getElementById('backupHistoryContent');
        content.innerHTML = '<div style="text-align: center; padding: 2rem; color: rgba(255, 255, 255, 0.8);">در حال بارگذاری...</div>';
        
        // Simulate loading backup history
        setTimeout(() => {
            const backups = <?php echo json_encode($backups); ?>;
            let html = '<div style="display: grid; gap: 1rem;">';
            
            backups.forEach(backup => {
                const isAuto = backup.backup_name.includes('_auto');
                const status = isAuto ? '🔄 خودکار' : '👤 دستی';
                const statusColor = isAuto ? 'var(--accent-400)' : 'var(--primary-400)';
                
                html += `
                    <div class="history-item" style="background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(10px); border: 1px solid rgba(255, 255, 255, 0.2); border-radius: 8px; padding: 1rem; border-left: 4px solid ${statusColor}; transition: all 0.3s ease;">
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 0.5rem;">
                            <div style="font-weight: bold; color: white;">${backup.backup_name}</div>
                            <div style="font-size: 0.875rem; color: ${statusColor}; font-weight: bold;">${status}</div>
                        </div>
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 0.5rem; font-size: 0.875rem; color: rgba(255, 255, 255, 0.7);">
                            <div>📅 ${backup.created_at}</div>
                            <div>📦 ${backup.files_count} فایل</div>
                            <div>💾 ${backup.total_size}</div>
                        </div>
                    </div>
                `;
            });
            
            html += '</div>';
            content.innerHTML = html;
        }, 1000);
    }

    // Bulk selection functionality
    function toggleSelectAll() {
        const selectAll = document.getElementById('selectAll');
        const checkboxes = document.querySelectorAll('.backup-checkbox');
        const bulkActions = document.getElementById('bulkActions');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = selectAll.checked;
        });
        
        updateBulkActions();
    }

    function updateBulkActions() {
        const checkboxes = document.querySelectorAll('.backup-checkbox:checked');
        const bulkActions = document.getElementById('bulkActions');
        
        if (checkboxes.length > 0) {
            bulkActions.classList.add('show');
            document.getElementById('selectedCount').textContent = checkboxes.length;
        } else {
            bulkActions.classList.remove('show');
        }
    }

    function deleteSelectedBackups() {
        const checkboxes = document.querySelectorAll('.backup-checkbox:checked');
        const backupNames = Array.from(checkboxes).map(cb => cb.value);
        
        if (backupNames.length === 0) return;
        
        if (confirm(`آیا از حذف ${backupNames.length} بکاپ انتخاب شده اطمینان دارید؟`)) {
            // Create form and submit
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '';
            
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'delete_multiple_backups';
            form.appendChild(actionInput);
            
            backupNames.forEach(name => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'backup_names[]';
                input.value = name;
                form.appendChild(input);
            });
            
            document.body.appendChild(form);
            form.submit();
        }
    }

    // Initialize
    document.addEventListener('DOMContentLoaded', function() {
        // Add event listeners to backup forms
        document.querySelectorAll('.backup-form').forEach(form => {
            form.onsubmit = () => handleBackupSubmit(form);
        });
        
        // Add event listeners to checkboxes
        document.querySelectorAll('.backup-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', updateBulkActions);
        });
    });
    </script>

    <style>
        /* Responsive Design for Advanced Backup Panel */
        @media (max-width: 768px) {
            .card-body {
                padding: 1rem;
            }
            
            .card-body > div {
                grid-template-columns: 1fr !important;
            }
            
            .backup-actions {
                flex-direction: column;
            }
            
            .backup-actions .btn {
                width: 100%;
                margin-bottom: 0.5rem;
            }
            
            .backup-table {
                font-size: 0.875rem;
            }
            
            .backup-table th,
            .backup-table td {
                padding: 0.5rem 0.25rem;
            }
            
            .bulk-actions {
                width: 90%;
                left: 5%;
                transform: none;
            }
            
            .progress-content {
                width: 95%;
                padding: 1.5rem;
            }
            
            #backupProgressModal > div,
            #backupHistoryModal > div {
                width: 95%;
                padding: 1.5rem;
            }
        }

        @media (max-width: 480px) {
            .card-header h2 {
                font-size: 1.25rem;
            }
            
            .backup-table {
                font-size: 0.75rem;
            }
            
            .backup-table th,
            .backup-table td {
                padding: 0.25rem 0.125rem;
            }
            
            .backup-actions .btn {
                font-size: 0.75rem;
                padding: 0.25rem;
            }
            
            .empty-state {
                padding: 2rem 0.5rem;
            }
            
            .empty-state .icon {
                font-size: 3rem;
            }
        }

        /* Advanced Panel Specific Styles */
        .advanced-panel-card {
            background: rgba(255, 255, 255, 0.1) !important;
            backdrop-filter: blur(20px) !important;
            border: 1px solid rgba(255, 255, 255, 0.2) !important;
        }

        .stat-card {
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
            background: rgba(255, 255, 255, 0.15) !important;
        }

        .quick-action-btn {
            transition: all 0.3s ease;
            border: none;
            color: white;
            font-weight: 500;
        }

        .quick-action-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
        }

        .modal-backdrop {
            backdrop-filter: blur(10px);
        }

        .history-item {
            transition: all 0.3s ease;
        }

        .history-item:hover {
            transform: translateX(-5px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
            background: rgba(255, 255, 255, 0.15) !important;
        }

        /* Glassmorphism effects */
        .glassmorphism-card {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .glassmorphism-text {
            color: white;
        }

        .glassmorphism-text-secondary {
            color: rgba(255, 255, 255, 0.7);
        }
    </style>
</body>
</html>
