<?php
/**
 * Automatic Backup Script
 * This script should be run by cron job to create automatic backups
 * 
 * Example cron job (daily at 2 AM):
 * 0 2 * * * /usr/bin/php /path/to/your/backend/cron/auto-backup.php
 */

// Set error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Set time limit for long operations
set_time_limit(3600); // 1 hour

// Set timezone
date_default_timezone_set('Asia/Tehran');



// Include required files
$dbFile = null;
$funcFile = null;

// پیدا کردن فایل database.php
foreach (['../config/database.php', dirname(__DIR__) . '/config/database.php'] as $path) {
    if (file_exists($path)) {
        $dbFile = $path;
        break;
    }
}

// پیدا کردن فایل functions.php
foreach (['../includes/functions.php', dirname(__DIR__) . '/includes/functions.php'] as $path) {
    if (file_exists($path)) {
        $funcFile = $path;
        break;
    }
}

if (!$dbFile || !$funcFile) {
    exit(1);
}

require_once $dbFile;
require_once $funcFile;

// Log function for cron
function logCron($message) {
    $logFile = '../logs/cron-backup.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[{$timestamp}] {$message}\n";
    

    
    // سعی کن در فایل لاگ بنویسد
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        if (!@mkdir($logDir, 0755, true)) {
            // اگر نتوانست پوشه ایجاد کند، لاگ را در پوشه فعلی بنویسد
            $logFile = './cron-backup.log';
        }
    }
    
    @file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    logCron("Starting automatic backup process...");
    
    // Check if automatic backup is enabled
    $backupEnabled = getSystemSetting('backup_enabled');
    logCron("Backup Enabled: " . ($backupEnabled ?: 'Not set'));
    
    if ($backupEnabled !== '1') {
        logCron("Automatic backup is disabled. Exiting.");
        exit(0);
    }
    
    // Check if backup should run based on frequency
    $shouldRun = shouldRunAutomaticBackup();
    logCron("Should run backup: " . ($shouldRun ? 'Yes' : 'No'));
    
    if (!$shouldRun) {
        logCron("Backup not due yet. Exiting.");
        exit(0);
    }
    
    logCron("Creating automatic backup...");
    
    // Create backup
    $result = createBackup('full');
    
    if ($result['success']) {
        logCron("Automatic backup completed successfully: " . $result['backup_name']);
        
        // Get backup info
        if (isset($result['info'])) {
            $info = $result['info'];
            logCron("Backup details: " . $info['files_count'] . " files, " . formatFileSize($info['total_size']) . " total size");
        }
        
        // Clean old backups (keep last 5)
        if (function_exists('cleanOldBackups')) {
            cleanOldBackups(5);
        }
        
        logCron("Automatic backup process completed successfully.");
        exit(0);
    } else {
        logCron("Automatic backup failed: " . $result['error']);
        exit(1);
    }
    
} catch (Exception $e) {
    logCron("Critical error in automatic backup: " . $e->getMessage());
    logCron("Stack trace: " . $e->getTraceAsString());
    exit(1);
}

/**
 * Clean old automatic backups, keeping only the specified number of recent ones
 */
function cleanOldBackups($keepCount = 5) {
    try {
        $backups = getBackups();
        
        // Filter only automatic backups (created by cron)
        $automaticBackups = array_filter($backups, function($backup) {
            // Check if backup name contains date format (automatic backups)
            return preg_match('/^\d{4}-\d{2}-\d{2}_\d{2}-\d{2}-\d{2}_auto/', $backup['backup_name']);
        });
        
        if (count($automaticBackups) <= $keepCount) {
            logCron("No old automatic backups to clean. Keeping " . count($automaticBackups) . " automatic backups.");
            return;
        }
        
        // Sort by creation date (newest first)
        usort($automaticBackups, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        // Get automatic backups to delete (keep the first $keepCount)
        $backupsToDelete = array_slice($automaticBackups, $keepCount);
        
        $deletedCount = 0;
        foreach ($backupsToDelete as $backup) {
            if (deleteBackup($backup['backup_name'])) {
                $deletedCount++;
                logCron("Deleted old automatic backup: " . $backup['backup_name']);
            } else {
                logCron("Failed to delete old automatic backup: " . $backup['backup_name']);
            }
        }
        
        logCron("Cleaned {$deletedCount} old automatic backups. Keeping {$keepCount} recent automatic backups.");
        
    } catch (Exception $e) {
        logCron("Error cleaning old automatic backups: " . $e->getMessage());
    }
}


?>
