<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to feature-requests.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        switch ($action) {
            case 'update_status':
                $requestId = (int)($_POST['request_id'] ?? 0);
                $status = sanitizeInput($_POST['status'] ?? '');
                $priority = sanitizeInput($_POST['priority'] ?? '');
                
                if ($requestId && $status) {
                    $updateQuery = "UPDATE feature_requests SET status = ?, priority = ?, updated_at = NOW() WHERE id = ?";
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->execute([$status, $priority, $requestId]);
                    
                    $message = "وضعیت درخواست ویژگی با موفقیت بروزرسانی شد";
                    logSecurityEvent('feature_status_updated', 'Request ID: ' . $requestId . ', Status: ' . $status, 'low');
                }
                break;
                
            case 'bulk_update':
                $selectedRequests = $_POST['selected_requests'] ?? [];
                $bulkStatus = sanitizeInput($_POST['bulk_status'] ?? '');
                $bulkPriority = sanitizeInput($_POST['bulk_priority'] ?? '');
                
                if (!empty($selectedRequests) && $bulkStatus) {
                    $updateQuery = "UPDATE feature_requests SET status = ?, priority = ?, updated_at = NOW() WHERE id IN (" . str_repeat('?,', count($selectedRequests) - 1) . "?)";
                    $params = array_merge([$bulkStatus, $bulkPriority], $selectedRequests);
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->execute($params);
                    
                    $message = count($selectedRequests) . " درخواست ویژگی با موفقیت بروزرسانی شد";
                    logSecurityEvent('features_bulk_updated', 'Count: ' . count($selectedRequests), 'low');
                }
                break;
                
            case 'delete':
                $requestId = (int)($_POST['request_id'] ?? 0);
                
                if ($requestId) {
                    $deleteQuery = "DELETE FROM feature_requests WHERE id = ?";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute([$requestId]);
                    
                    $message = "درخواست ویژگی با موفقیت حذف شد";
                    logSecurityEvent('feature_deleted', 'Request ID: ' . $requestId, 'medium');
                }
                break;
                
            case 'bulk_delete':
                $selectedRequests = $_POST['selected_requests'] ?? [];
                
                if (!empty($selectedRequests)) {
                    $deleteQuery = "DELETE FROM feature_requests WHERE id IN (" . str_repeat('?,', count($selectedRequests) - 1) . "?)";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute($selectedRequests);
                    
                    $message = count($selectedRequests) . " درخواست ویژگی با موفقیت حذف شد";
                    logSecurityEvent('features_bulk_deleted', 'Count: ' . count($selectedRequests), 'medium');
                }
                break;
                
            case 'get_request_details':
                $requestId = (int)($_POST['request_id'] ?? 0);
                
                if ($requestId) {
                    $requestQuery = "SELECT * FROM feature_requests WHERE id = ?";
                    $requestStmt = $conn->prepare($requestQuery);
                    $requestStmt->execute([$requestId]);
                    $request = $requestStmt->fetch();
                    
                    if ($request) {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => true,
                            'request' => [
                                'id' => $request['id'],
                                'feature' => htmlspecialchars($request['feature']),
                                'description' => htmlspecialchars($request['description']),
                                'name' => htmlspecialchars($request['name']),
                                'email' => htmlspecialchars($request['email']),
                                'status' => $request['status'],
                                'priority' => $request['priority'],
                                'created_at' => $request['created_at']
                            ]
                        ]);
                    } else {
                        header('Content-Type: application/json');
                        echo json_encode(['success' => false, 'message' => 'درخواست ویژگی یافت نشد']);
                    }
                } else {
                    header('Content-Type: application/json');
                    echo json_encode(['success' => false, 'message' => 'شناسه درخواست نامعتبر است']);
                }
                exit;
                break;
        }
    } catch (Exception $e) {
        error_log("Feature requests error: " . $e->getMessage());
        $error = 'خطا در عملیات. لطفاً دوباره تلاش کنید';
    }
}

// Get feature requests with filtering
$statusFilter = $_GET['status'] ?? '';
$priorityFilter = $_GET['priority'] ?? '';
$searchQuery = $_GET['search'] ?? '';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $whereConditions = [];
    $params = [];
    
    if ($statusFilter) {
        $whereConditions[] = "status = ?";
        $params[] = $statusFilter;
    }
    
    if ($priorityFilter) {
        $whereConditions[] = "priority = ?";
        $params[] = $priorityFilter;
    }
    
    if ($searchQuery) {
        $whereConditions[] = "(feature LIKE ? OR description LIKE ? OR name LIKE ? OR email LIKE ?)";
        $searchParam = '%' . $searchQuery . '%';
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
    }
    
    $whereClause = '';
    if (!empty($whereConditions)) {
        $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
    }
    
    // Check if new columns exist, otherwise use basic query
    try {
        $checkColumnsQuery = "SHOW COLUMNS FROM feature_requests LIKE 'attachment_path'";
        $checkStmt = $conn->query($checkColumnsQuery);
        $hasNewColumns = $checkStmt->rowCount() > 0;
        $checkStmt->closeCursor();
    } catch (Exception $e) {
        $hasNewColumns = false;
    }
    
    if ($hasNewColumns) {
        $requestsQuery = "SELECT id, name, email, feature, description, priority, status, ip_address, user_agent, attachment_path, admin_response, response_sent_at, created_at, updated_at FROM feature_requests $whereClause ORDER BY created_at DESC";
    } else {
        $requestsQuery = "SELECT id, name, email, feature, description, priority, status, ip_address, user_agent, created_at, updated_at FROM feature_requests $whereClause ORDER BY created_at DESC";
    }
    
    $requestsStmt = $conn->prepare($requestsQuery);
    $requestsStmt->execute($params);
    $requests = $requestsStmt->fetchAll();
    
    // Add default values for new columns if they don't exist
    if (!$hasNewColumns) {
        foreach ($requests as &$request) {
            $request['attachment_path'] = null;
            $request['admin_response'] = null;
            $request['response_sent_at'] = null;
        }
        unset($request);
    }
    
    // Get statistics
    $statsQuery = "SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'new' THEN 1 ELSE 0 END) as new_count,
        SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress_count,
        SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_count,
        SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_count,
        SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_count
        FROM feature_requests";
    $statsStmt = $conn->query($statsQuery);
    $stats = $statsStmt->fetch();
    
} catch (Exception $e) {
    error_log("Feature requests list error: " . $e->getMessage());
    $error = 'خطا در بارگذاری درخواست‌های ویژگی';
    $requests = [];
    $stats = ['total' => 0, 'new_count' => 0, 'in_progress_count' => 0, 'approved_count' => 0, 'rejected_count' => 0, 'completed_count' => 0];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>درخواست‌های ویژگی - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    <?php include 'includes/admin-styles.php'; ?>
    <style>
        /* Notification System */
        .notification-container {
            position: fixed;
            top: 20px;
            left: 20px;
            z-index: 10000;
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .notification {
            min-width: 300px;
            max-width: 500px;
            padding: 1rem 1.5rem;
            border-radius: 12px;
            box-shadow: 0 8px 24px rgba(0, 0, 0, 0.3);
            display: flex;
            align-items: center;
            gap: 1rem;
            animation: slideInLeft 0.3s ease-out;
            font-family: 'Vazirmatn', 'Tahoma', Arial, sans-serif;
        }
        
        .notification-success {
            background: linear-gradient(135deg, #27ae60, #2ecc71);
            color: white;
            border: 2px solid rgba(255, 255, 255, 0.2);
        }
        
        .notification-error {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            color: white;
            border: 2px solid rgba(255, 255, 255, 0.2);
        }
        
        .notification-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            color: white;
            border: 2px solid rgba(255, 255, 255, 0.2);
        }
        
        .notification-icon {
            font-size: 1.5rem;
            flex-shrink: 0;
        }
        
        .notification-content {
            flex: 1;
        }
        
        .notification-title {
            font-weight: 700;
            font-size: 1rem;
            margin-bottom: 0.25rem;
        }
        
        .notification-message {
            font-size: 0.875rem;
            opacity: 0.95;
            line-height: 1.5;
        }
        
        .notification-close {
            background: rgba(255, 255, 255, 0.2);
            border: none;
            color: white;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
            flex-shrink: 0;
            transition: background 0.2s;
        }
        
        .notification-close:hover {
            background: rgba(255, 255, 255, 0.3);
        }
        
        @keyframes slideInLeft {
            from {
                transform: translateX(-100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        
        @keyframes slideOutLeft {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(-100%);
                opacity: 0;
            }
        }
        
        .notification.hiding {
            animation: slideOutLeft 0.3s ease-out forwards;
        }
    </style>
</head>
<body>
    <!-- Notification Container -->
    <div id="notificationContainer" class="notification-container"></div>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">✨ درخواست‌های ویژگی</h1>
            <p class="page-description">مدیریت و پیگیری درخواست‌های ویژگی جدید</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="grid grid-cols-1 grid-cols-2 grid-cols-3 grid-cols-4" style="margin-bottom: 2rem;">
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['total']; ?></div>
                <div class="stats-label">کل درخواست‌ها</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['new_count']; ?></div>
                <div class="stats-label">جدید</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['in_progress_count']; ?></div>
                <div class="stats-label">در حال بررسی</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['approved_count']; ?></div>
                <div class="stats-label">تایید شده</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['rejected_count']; ?></div>
                <div class="stats-label">رد شده</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['completed_count']; ?></div>
                <div class="stats-label">تکمیل شده</div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">🔍 فیلترها</h2>
            </div>
            
            <form method="GET" action="" style="display: flex; gap: 1rem; align-items: end; flex-wrap: wrap;">
                <div class="form-group" style="flex: 1; min-width: 200px;">
                    <label for="search" class="form-label">جستجو</label>
                    <input type="text" id="search" name="search" class="form-input" 
                           value="<?php echo htmlspecialchars($searchQuery); ?>" 
                           placeholder="جستجو در ویژگی، توضیحات، نام یا ایمیل">
                </div>
                
                <div class="form-group" style="min-width: 150px;">
                    <label for="status" class="form-label">وضعیت</label>
                    <select id="status" name="status" class="form-select">
                        <option value="">همه</option>
                        <option value="new" <?php echo $statusFilter === 'new' ? 'selected' : ''; ?>>جدید</option>
                        <option value="in_progress" <?php echo $statusFilter === 'in_progress' ? 'selected' : ''; ?>>در حال بررسی</option>
                        <option value="approved" <?php echo $statusFilter === 'approved' ? 'selected' : ''; ?>>تایید شده</option>
                        <option value="rejected" <?php echo $statusFilter === 'rejected' ? 'selected' : ''; ?>>رد شده</option>
                        <option value="completed" <?php echo $statusFilter === 'completed' ? 'selected' : ''; ?>>تکمیل شده</option>
                    </select>
                </div>
                
                <div class="form-group" style="min-width: 150px;">
                    <label for="priority" class="form-label">اولویت</label>
                    <select id="priority" name="priority" class="form-select">
                        <option value="">همه</option>
                        <option value="low" <?php echo $priorityFilter === 'low' ? 'selected' : ''; ?>>کم</option>
                        <option value="medium" <?php echo $priorityFilter === 'medium' ? 'selected' : ''; ?>>متوسط</option>
                        <option value="high" <?php echo $priorityFilter === 'high' ? 'selected' : ''; ?>>زیاد</option>
                        <option value="critical" <?php echo $priorityFilter === 'critical' ? 'selected' : ''; ?>>بحرانی</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn btn-primary">اعمال فیلتر</button>
                    <a href="feature-requests.php" class="btn btn-secondary">پاک کردن</a>
                </div>
            </form>
        </div>

        <!-- Feature Requests List -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📋 لیست درخواست‌های ویژگی</h2>
                <div style="display: flex; gap: 0.5rem;">
                    <button type="button" class="btn btn-warning btn-sm" onclick="showBulkUpdateModal()">
                        بروزرسانی دسته‌ای
                    </button>
                    <button type="button" class="btn btn-danger btn-sm" onclick="deleteSelectedRequests()">
                        حذف انتخاب شده‌ها
                    </button>
                </div>
            </div>
            
            <?php if (!empty($requests)): ?>
                <form method="POST" action="" id="requestsForm">
                    <input type="hidden" name="action" value="bulk_delete">
                    
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>
                                        <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                                    </th>
                                    <th>ویژگی</th>
                                    <th>درخواست‌کننده</th>
                                    <th>اولویت</th>
                                    <th>وضعیت</th>
                                    <th>تاریخ</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($requests as $request): ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" name="selected_requests[]" value="<?php echo $request['id']; ?>" class="request-checkbox">
                                        </td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($request['feature']); ?></strong>
                                            <br>
                                            <small style="color: var(--dark-400);">
                                                <?php echo htmlspecialchars(substr($request['description'], 0, 100)) . (strlen($request['description']) > 100 ? '...' : ''); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <div><?php echo htmlspecialchars($request['name']); ?></div>
                                            <small style="color: var(--dark-400);"><?php echo htmlspecialchars($request['email']); ?></small>
                                        </td>
                                        <td><?php echo getPriorityBadge($request['priority']); ?></td>
                                        <td><?php echo getStatusBadge($request['status']); ?></td>
                                        <td><?php echo formatDate($request['created_at']); ?></td>
                                        <td>
                                            <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                                                <button type="button" class="btn btn-warning btn-sm" onclick="editRequest(<?php echo $request['id']; ?>, '<?php echo htmlspecialchars($request['feature']); ?>', '<?php echo $request['status']; ?>', '<?php echo $request['priority']; ?>')">
                                                    ویرایش
                                                </button>
                                                <button type="button" class="btn btn-info btn-sm" onclick="viewRequestAjax(<?php echo $request['id']; ?>)">
                                                    مشاهده
                                                </button>
                                                <button type="button" class="btn btn-primary btn-sm" onclick="showResponseModal(<?php echo $request['id']; ?>, '<?php echo htmlspecialchars($request['feature']); ?>', '<?php echo htmlspecialchars($request['email']); ?>', '<?php echo htmlspecialchars($request['name']); ?>', <?php echo $request['admin_response'] ? 'true' : 'false'; ?>, '<?php echo htmlspecialchars($request['admin_response'] ?? ''); ?>')">
                                                    <?php echo $request['admin_response'] ? '✏️ ویرایش پاسخ' : '💬 پاسخ'; ?>
                                                </button>
                                                <button type="button" class="btn btn-danger btn-sm" onclick="deleteRequest(<?php echo $request['id']; ?>)">
                                                    حذف
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
            <?php else: ?>
                <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هیچ درخواست ویژگی‌ای یافت نشد</p>
            <?php endif; ?>
        </div>
    </div>

    <!-- Edit Request Modal -->
    <div id="editModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 400px;">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400);">ویرایش درخواست ویژگی</h3>
            
            <form method="POST" action="" id="editRequestForm">
                <input type="hidden" name="action" value="update_status">
                <input type="hidden" name="request_id" id="editRequestId">
                
                <div class="form-group">
                    <label for="editFeature" class="form-label">ویژگی</label>
                    <input type="text" id="editFeature" class="form-input" readonly>
                </div>
                
                <div class="form-group">
                    <label for="editStatus" class="form-label">وضعیت</label>
                    <select id="editStatus" name="status" class="form-select">
                        <option value="new">جدید</option>
                        <option value="in_progress">در حال بررسی</option>
                        <option value="approved">تایید شده</option>
                        <option value="rejected">رد شده</option>
                        <option value="completed">تکمیل شده</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="editPriority" class="form-label">اولویت</label>
                    <select id="editPriority" name="priority" class="form-select">
                        <option value="low">کم</option>
                        <option value="medium">متوسط</option>
                        <option value="high">زیاد</option>
                        <option value="critical">بحرانی</option>
                    </select>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1rem;">
                    <button type="submit" class="btn btn-primary">ذخیره تغییرات</button>
                    <button type="button" class="btn btn-secondary" onclick="closeEditModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <!-- View Request Modal -->
    <div id="viewModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 500px; max-width: 90%; max-height: 90%; display: flex; flex-direction: column; box-shadow: 0 20px 40px rgba(0,0,0,0.3);">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400); flex-shrink: 0;">مشاهده درخواست ویژگی</h3>
            
            <div id="viewRequestContent" style="flex: 1; overflow-y: auto; padding-right: 0.5rem; scrollbar-width: thin; scrollbar-color: var(--dark-600) var(--dark-800);">
                <!-- Content will be filled by JavaScript -->
            </div>
            
            <div style="display: flex; gap: 1rem; margin-top: 1rem; flex-shrink: 0;">
                <button type="button" class="btn btn-secondary" onclick="closeViewModal()">بستن</button>
            </div>
        </div>
    </div>

    <!-- Response Modal -->
    <div id="responseModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 500px; max-width: 90%; max-height: 90%; display: flex; flex-direction: column; box-shadow: 0 20px 40px rgba(0,0,0,0.3);">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400); flex-shrink: 0;">💬 ارسال پاسخ</h3>
            
            <div id="responseRequestInfo" style="margin-bottom: 1rem; padding: 1rem; background: var(--dark-700); border-radius: 0.5rem; flex-shrink: 0;">
                <!-- Request info will be filled by JavaScript -->
            </div>
            
            <form id="responseForm" style="flex: 1; display: flex; flex-direction: column;">
                <input type="hidden" id="responseRequestId" value="">
                <input type="hidden" id="responseFormType" value="feature_request">
                
                <div class="form-group" style="flex: 1; display: flex; flex-direction: column;">
                    <label for="responseText" class="form-label">متن پاسخ *</label>
                    <textarea id="responseText" class="form-input" rows="10" required style="flex: 1; resize: none;" placeholder="پاسخ خود را اینجا بنویسید..."></textarea>
                </div>
                
                <div id="existingResponse" style="margin-top: 1rem; padding: 1rem; background: rgba(197, 145, 44, 0.1); border: 1px solid rgba(197, 145, 44, 0.3); border-radius: 0.5rem; display: none;">
                    <p style="color: var(--secondary-color); font-weight: 600; margin-bottom: 0.5rem;">📝 پاسخ قبلی:</p>
                    <p id="existingResponseText" style="color: var(--dark-300); white-space: pre-wrap;"></p>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1rem; flex-shrink: 0;">
                    <button type="submit" class="btn btn-primary" id="responseSubmitBtn">📧 ارسال پاسخ</button>
                    <button type="button" class="btn btn-secondary" onclick="closeResponseModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Bulk Update Modal -->
    <div id="bulkUpdateModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 400px;">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400);">بروزرسانی دسته‌ای</h3>
            <p id="bulkUpdateCount" style="margin-bottom: 1rem; color: var(--dark-300);"></p>
            
            <form method="POST" action="" id="bulkUpdateForm">
                <input type="hidden" name="action" value="bulk_update">
                <div id="bulkUpdateSelectedRequests">
                    <!-- Selected request IDs will be added here dynamically -->
                </div>
                
                <div class="form-group">
                    <label for="bulkStatus" class="form-label">وضعیت جدید</label>
                    <select id="bulkStatus" name="bulk_status" class="form-select">
                        <option value="">انتخاب کنید</option>
                        <option value="new">جدید</option>
                        <option value="in_progress">در حال بررسی</option>
                        <option value="approved">تایید شده</option>
                        <option value="rejected">رد شده</option>
                        <option value="completed">تکمیل شده</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="bulkPriority" class="form-label">اولویت جدید</label>
                    <select id="bulkPriority" name="bulk_priority" class="form-select">
                        <option value="">انتخاب کنید</option>
                        <option value="low">کم</option>
                        <option value="medium">متوسط</option>
                        <option value="high">زیاد</option>
                        <option value="critical">بحرانی</option>
                    </select>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1rem;">
                    <button type="submit" class="btn btn-primary">بروزرسانی</button>
                    <button type="button" class="btn btn-secondary" onclick="closeBulkUpdateModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Toggle select all checkbox
        function toggleSelectAll() {
            const selectAll = document.getElementById('selectAll');
            const checkboxes = document.querySelectorAll('.request-checkbox');
            
            checkboxes.forEach(checkbox => {
                checkbox.checked = selectAll.checked;
            });
        }
        
        // Delete selected requests
        function deleteSelectedRequests() {
            const checkboxes = document.querySelectorAll('.request-checkbox:checked');
            
            if (checkboxes.length === 0) {
                alert('لطفاً حداقل یک درخواست را انتخاب کنید');
                return;
            }
            
            if (confirm('آیا از حذف ' + checkboxes.length + ' درخواست انتخاب شده اطمینان دارید؟')) {
                document.getElementById('requestsForm').submit();
            }
        }
        
        // Edit request modal
        function editRequest(requestId, feature, status, priority) {
            document.getElementById('editRequestId').value = requestId;
            document.getElementById('editFeature').value = feature;
            document.getElementById('editStatus').value = status;
            document.getElementById('editPriority').value = priority;
            document.getElementById('editModal').style.display = 'block';
        }
        
        function closeEditModal() {
            document.getElementById('editModal').style.display = 'none';
        }
        
        // View request modal using AJAX
        function viewRequestAjax(requestId) {
            // Show loading state
            document.getElementById('viewRequestContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">در حال بارگذاری...</div>';
            document.getElementById('viewModal').style.display = 'block';
            
            // Create form data
            const formData = new FormData();
            formData.append('action', 'get_request_details');
            formData.append('request_id', requestId);
            
            // Send AJAX request
            fetch('feature-requests.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(data => {
                try {
                    const requestData = JSON.parse(data);
                    if (requestData.success) {
                        const content = `
                            <div class="form-group">
                                <label class="form-label">ویژگی</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                                    ${requestData.request.feature}
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">توضیحات</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); white-space: pre-wrap; word-wrap: break-word; overflow-wrap: break-word; line-height: 1.6; max-height: none; min-height: 100px;">
                                    ${requestData.request.description}
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">درخواست‌کننده</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                                    ${requestData.request.name} (${requestData.request.email})
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">شناسه درخواست</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600);">
                                    #${requestData.request.id}
                                </div>
                            </div>
                        `;
                        document.getElementById('viewRequestContent').innerHTML = content;
                    } else {
                        document.getElementById('viewRequestContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در بارگذاری اطلاعات</div>';
                    }
                } catch (e) {
                    document.getElementById('viewRequestContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در پردازش اطلاعات</div>';
                }
            })
            .catch(error => {
                document.getElementById('viewRequestContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در ارتباط با سرور</div>';
            });
        }
        
        // Legacy view request modal (for backward compatibility)
        function viewRequest(requestId, feature, description, name, email) {
            // This function is kept for backward compatibility
            const content = `
                <div class="form-group">
                    <label class="form-label">ویژگی</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                        ${feature}
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">توضیحات</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); white-space: pre-wrap; word-wrap: break-word; overflow-wrap: break-word; line-height: 1.6; max-height: none; min-height: 100px;">
                        ${description}
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">درخواست‌کننده</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                        ${name} (${email})
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">شناسه درخواست</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600);">
                        #${requestId}
                    </div>
                </div>
            `;
            
            document.getElementById('viewRequestContent').innerHTML = content;
            document.getElementById('viewModal').style.display = 'block';
        }
        
        function closeViewModal() {
            document.getElementById('viewModal').style.display = 'none';
        }
        
        // Bulk update modal
        function showBulkUpdateModal() {
            const checkboxes = document.querySelectorAll('.request-checkbox:checked');
            
            if (checkboxes.length === 0) {
                alert('لطفاً حداقل یک درخواست را انتخاب کنید');
                return;
            }
            
            // Clear previous form data
            document.getElementById('bulkStatus').value = '';
            document.getElementById('bulkPriority').value = '';
            
            // Add selected request IDs to the form
            const selectedRequestsContainer = document.getElementById('bulkUpdateSelectedRequests');
            selectedRequestsContainer.innerHTML = '';
            
            checkboxes.forEach(checkbox => {
                const hiddenInput = document.createElement('input');
                hiddenInput.type = 'hidden';
                hiddenInput.name = 'selected_requests[]';
                hiddenInput.value = checkbox.value;
                selectedRequestsContainer.appendChild(hiddenInput);
            });
            
            // Show count of selected items
            document.getElementById('bulkUpdateCount').textContent = `${checkboxes.length} درخواست انتخاب شده است`;
            
            document.getElementById('bulkUpdateModal').style.display = 'block';
        }
        
        function closeBulkUpdateModal() {
            document.getElementById('bulkUpdateModal').style.display = 'none';
        }
        
        // Response modal functions
        function showResponseModal(requestId, feature, email, name, hasResponse, existingResponse) {
            document.getElementById('responseRequestId').value = requestId;
            document.getElementById('responseText').value = '';
            
            const requestInfo = `
                <div style="margin-bottom: 0.5rem;">
                    <strong style="color: var(--primary-400);">💡 ویژگی:</strong>
                    <span style="color: var(--dark-300);">${feature}</span>
                </div>
                <div style="margin-bottom: 0.5rem;">
                    <strong style="color: var(--primary-400);">👤 نام:</strong>
                    <span style="color: var(--dark-300);">${name}</span>
                </div>
                <div>
                    <strong style="color: var(--primary-400);">📧 ایمیل:</strong>
                    <span style="color: var(--dark-300);">${email}</span>
                </div>
            `;
            document.getElementById('responseRequestInfo').innerHTML = requestInfo;
            
            if (hasResponse && existingResponse) {
                document.getElementById('existingResponse').style.display = 'block';
                document.getElementById('existingResponseText').textContent = existingResponse;
            } else {
                document.getElementById('existingResponse').style.display = 'none';
            }
            
            document.getElementById('responseModal').style.display = 'block';
        }
        
        function closeResponseModal() {
            document.getElementById('responseModal').style.display = 'none';
            document.getElementById('responseForm').reset();
        }
        
        // Handle response form submission
        document.getElementById('responseForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const requestId = document.getElementById('responseRequestId').value;
            const response = document.getElementById('responseText').value;
            const formType = document.getElementById('responseFormType').value;
            const submitBtn = document.getElementById('responseSubmitBtn');
            
            if (!response.trim()) {
                alert('لطفاً متن پاسخ را وارد کنید');
                return;
            }
            
            submitBtn.disabled = true;
            submitBtn.textContent = 'در حال ارسال...';
            
            try {
                const responseData = await fetch('api/send-response.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        form_type: formType,
                        id: requestId,
                        response: response
                    })
                });
                
                if (!responseData.ok) {
                    throw new Error('HTTP error! status: ' + responseData.status);
                }
                
                const data = await responseData.json();
                
                if (data.success) {
                    const message = 'پاسخ با موفقیت ارسال شد' + (data.email_sent ? ' و ایمیل به کاربر ارسال شد' : '');
                    showNotification('success', '✅ موفق', message);
                    closeResponseModal();
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    showNotification('error', '❌ خطا', 'خطا در ارسال پاسخ: ' + (data.message || 'خطای نامشخص'));
                }
            } catch (error) {
                console.error('Error sending response:', error);
                showNotification('error', '❌ خطا', 'خطا در اتصال به سرور: ' + error.message);
            } finally {
                submitBtn.disabled = false;
                submitBtn.textContent = '📧 ارسال پاسخ';
            }
        });
        
        // Delete single request
        function deleteRequest(requestId) {
            if (confirm('آیا از حذف این درخواست ویژگی اطمینان دارید؟')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="request_id" value="${requestId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        // Close modals when clicking outside
        window.onclick = function(event) {
            const editModal = document.getElementById('editModal');
            const viewModal = document.getElementById('viewModal');
            const bulkUpdateModal = document.getElementById('bulkUpdateModal');
            
            if (event.target === editModal) {
                closeEditModal();
            }
            if (event.target === viewModal) {
                closeViewModal();
            }
            if (event.target === bulkUpdateModal) {
                closeBulkUpdateModal();
            }
            const responseModal = document.getElementById('responseModal');
            if (event.target === responseModal) {
                closeResponseModal();
            }
        }
        
        // Close modals when pressing Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeEditModal();
                closeViewModal();
                closeBulkUpdateModal();
                closeResponseModal();
            }
        });
        
        // Notification System
        function showNotification(type, title, message) {
            const container = document.getElementById('notificationContainer');
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;
            
            const icon = type === 'success' ? '✅' : type === 'error' ? '❌' : '⚠️';
            
            notification.innerHTML = `
                <span class="notification-icon">${icon}</span>
                <div class="notification-content">
                    <div class="notification-title">${title}</div>
                    <div class="notification-message">${message}</div>
                </div>
                <button class="notification-close" onclick="this.parentElement.remove()">×</button>
            `;
            
            container.appendChild(notification);
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                notification.classList.add('hiding');
                setTimeout(() => {
                    if (notification.parentElement) {
                        notification.remove();
                    }
                }, 300);
            }, 5000);
        }
        
        // Form validation
        document.getElementById('bulkUpdateForm').addEventListener('submit', function(e) {
            const status = document.getElementById('bulkStatus').value;
            const priority = document.getElementById('bulkPriority').value;
            
            if (!status && !priority) {
                e.preventDefault();
                alert('لطفاً حداقل یک فیلد را انتخاب کنید');
                return false;
            }
            
            // Show confirmation
            const checkboxes = document.querySelectorAll('.request-checkbox:checked');
            if (!confirm(`آیا از بروزرسانی ${checkboxes.length} درخواست انتخاب شده اطمینان دارید؟`)) {
                e.preventDefault();
                return false;
            }
        });
    </script>
</body>
</html>
