<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once 'config/database.php';
require_once 'includes/functions.php';
require_once 'includes/ip-geolocation.php';

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['ip_address'])) {
        throw new Exception('آدرس IP الزامی است');
    }
    
    $ipAddress = $input['ip_address'];
    
    // Validate IP address
    if (!filter_var($ipAddress, FILTER_VALIDATE_IP)) {
        throw new Exception('آدرس IP نامعتبر است');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get IP activity data
    $activityStmt = $conn->prepare("
        SELECT 
            COUNT(*) as threat_count,
            MAX(created_at) as last_activity,
            GROUP_CONCAT(DISTINCT event_type ORDER BY event_type SEPARATOR ', ') as threat_types,
            GROUP_CONCAT(DISTINCT severity ORDER BY severity SEPARATOR ', ') as severities
        FROM security_logs 
        WHERE ip_address = ?
    ");
    $activityStmt->execute([$ipAddress]);
    $activityData = $activityStmt->fetch(PDO::FETCH_ASSOC);
    
    // Calculate risk score
    $riskScore = 0;
    $severityCounts = [
        'critical' => 0,
        'high' => 0,
        'medium' => 0,
        'low' => 0
    ];
    
    $severityStmt = $conn->prepare("
        SELECT severity, COUNT(*) as count
        FROM security_logs 
        WHERE ip_address = ?
        GROUP BY severity
    ");
    $severityStmt->execute([$ipAddress]);
    
    while ($row = $severityStmt->fetch(PDO::FETCH_ASSOC)) {
        $severityCounts[$row['severity']] = $row['count'];
    }
    
    // Calculate risk score based on severity
    $riskScore += $severityCounts['critical'] * 25;
    $riskScore += $severityCounts['high'] * 15;
    $riskScore += $severityCounts['medium'] * 8;
    $riskScore += $severityCounts['low'] * 3;
    
    // Cap risk score at 100
    $riskScore = min($riskScore, 100);
    
    // Get geographic information
    $geoData = getIPGeolocation($ipAddress);
    
    // Generate recommendations based on risk score
    $recommendations = [];
    if ($riskScore >= 80) {
        $recommendations[] = 'این IP خطر بالایی دارد و باید فوراً مسدود شود';
    } elseif ($riskScore >= 60) {
        $recommendations[] = 'نظارت دقیق بر فعالیت‌های این IP توصیه می‌شود';
    } elseif ($riskScore >= 40) {
        $recommendations[] = 'بررسی دوره‌ای فعالیت‌های این IP ضروری است';
    } else {
        $recommendations[] = 'این IP در حال حاضر خطری ندارد';
    }
    
    if ($activityData['threat_count'] > 10) {
        $recommendations[] = 'تعداد بالای تهدیدات نشان‌دهنده فعالیت مشکوک است';
    }
    
    if (strpos($activityData['threat_types'], 'brute_force') !== false) {
        $recommendations[] = 'تلاش‌های brute force شناسایی شده - تقویت رمز عبور ضروری است';
    }
    
    if (strpos($activityData['threat_types'], 'sql_injection') !== false) {
        $recommendations[] = 'تلاش‌های SQL Injection شناسایی شده - بررسی امنیت پایگاه داده ضروری است';
    }
    
    $response = [
        'success' => true,
        'risk_score' => $riskScore,
        'threat_count' => $activityData['threat_count'] ?? 0,
        'last_activity' => $activityData['last_activity'] ?? 'هیچ فعالیتی',
        'threat_types' => $activityData['threat_types'] ?? 'هیچ تهدیدی',
        'severities' => $activityData['severities'] ?? 'نامشخص',
        'country' => $geoData['country'] ?? 'نامشخص',
        'city' => $geoData['city'] ?? 'نامشخص',
        'isp' => $geoData['isp'] ?? 'نامشخص',
        'recommendations' => implode(' | ', $recommendations)
    ];
    
    echo json_encode($response);
    
} catch (Exception $e) {
    error_log("Error getting IP details: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'خطا در دریافت جزئیات IP: ' . $e->getMessage()
    ]);
}
?>
