<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Enhanced security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');
header('Content-Security-Policy: default-src \'self\'; script-src \'self\' \'unsafe-inline\'; style-src \'self\' \'unsafe-inline\' https://fonts.googleapis.com; font-src \'self\' https://fonts.gstatic.com; img-src \'self\' data:;');

// Start session with enhanced security
if (session_status() === PHP_SESSION_NONE) {
    // Configure session security
    ini_set('session.cookie_httponly', 1);
    ini_set('session.cookie_secure', isset($_SERVER['HTTPS']));
    ini_set('session.use_strict_mode', 1);
    ini_set('session.cookie_samesite', 'Strict');
    ini_set('session.gc_maxlifetime', 3600); // 1 hour
    ini_set('session.cookie_lifetime', 0); // Session cookie
    
    session_start();
    
    // Regenerate session ID on login page to prevent session fixation
    if (!isset($_SESSION['login_page_visited'])) {
        session_regenerate_id(true);
        $_SESSION['login_page_visited'] = true;
    }
}

// Include IP check system
require_once 'includes/ip-check.php';

require_once 'config/database.php';
require_once 'includes/functions.php';

// Generate CSRF token if not exists
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Check Admin IP Restriction
if (isAdminIPRestrictionEnabled()) {
    $clientIP = getClientIP();
    if (!isAllowedAdminIP($clientIP)) {
        // Log the blocked access attempt
        logSecurityEvent('admin_login_blocked_ip_restriction', 'Blocked IP tried to access admin login: ' . $clientIP, 'high', json_encode([
            'ip' => $clientIP,
            'user_agent' => getUserAgent()
        ]));
        
        // Show blocked page
        http_response_code(403);
        ?>
        <!DOCTYPE html>
        <html lang="fa" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>دسترسی مسدود شده - پنل ادمین</title>
            <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body {
                    font-family: 'Vazirmatn', 'Tahoma', Arial, sans-serif;
                    background: linear-gradient(135deg, #e74c3c 0%, #c0392b 100%);
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    color: white;
                    padding: 1rem;
                }
                .container {
                    background: rgba(0, 0, 0, 0.4);
                    border-radius: 2rem;
                    backdrop-filter: blur(20px);
                    border: 1px solid rgba(255, 255, 255, 0.1);
                    box-shadow: 0 25px 50px rgba(0, 0, 0, 0.3);
                    padding: 3rem 2rem;
                    max-width: 500px;
                    width: 100%;
                    text-align: center;
                }
                .icon { font-size: 4rem; margin-bottom: 1.5rem; }
                h1 { font-size: 2rem; margin-bottom: 1rem; color: #ffa726; font-weight: 700; }
                .description { font-size: 1.1rem; line-height: 1.6; margin-bottom: 2rem; opacity: 0.9; }
                .ip { background: rgba(255, 255, 255, 0.1); padding: 1rem; border-radius: 0.8rem; margin: 1rem 0; font-family: monospace; }
                .emergency-link { margin-top: 2rem; }
                .emergency-link a {
                    background: linear-gradient(45deg, #4fc3f7, #29b6f6);
                    color: white;
                    text-decoration: none;
                    padding: 1rem 2rem;
                    border-radius: 0.8rem;
                    font-weight: 500;
                    display: inline-block;
                    transition: all 0.3s ease;
                }
                .emergency-link a:hover { transform: translateY(-2px); box-shadow: 0 8px 16px rgba(0, 0, 0, 0.2); }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="icon">🚫</div>
                <h1>دسترسی مسدود شده</h1>
                <div class="description">
                    دسترسی شما به پنل ادمین مسدود شده است.<br>
                    فقط IP های مجاز می‌توانند به پنل دسترسی داشته باشند.
                </div>
                
                <div class="ip">
                    IP شما: <?php echo htmlspecialchars($clientIP); ?>
                </div>
            </div>
        </body>
        </html>
        <?php
        exit();
    }
}

// Redirect if already logged in
if (isLoggedIn()) {
    header('Location: dashboard.php');
    exit();
}

$error = '';
$success = '';

// Enhanced password strength validation function
function validatePasswordStrength($password) {
    $errors = [];
    
    if (strlen($password) < 8) {
        $errors[] = 'رمز عبور باید حداقل 8 کاراکتر باشد';
    }
    
    if (!preg_match('/[A-Z]/', $password)) {
        $errors[] = 'رمز عبور باید حداقل یک حرف بزرگ داشته باشد';
    }
    
    if (!preg_match('/[a-z]/', $password)) {
        $errors[] = 'رمز عبور باید حداقل یک حرف کوچک داشته باشد';
    }
    
    if (!preg_match('/[0-9]/', $password)) {
        $errors[] = 'رمز عبور باید حداقل یک عدد داشته باشد';
    }
    
    if (!preg_match('/[^A-Za-z0-9]/', $password)) {
        $errors[] = 'رمز عبور باید حداقل یک کاراکتر خاص داشته باشد';
    }
    
    return $errors;
}

// Enhanced input validation function
function validateLoginInput($username, $password) {
    $errors = [];
    
    // Username validation
    if (empty($username)) {
        $errors[] = 'نام کاربری نمی‌تواند خالی باشد';
    } elseif (strlen($username) < 3) {
        $errors[] = 'نام کاربری باید حداقل 3 کاراکتر باشد';
    } elseif (strlen($username) > 50) {
        $errors[] = 'نام کاربری نمی‌تواند بیش از 50 کاراکتر باشد';
    } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        $errors[] = 'نام کاربری فقط می‌تواند شامل حروف، اعداد و خط زیر باشد';
    }
    
    // Password validation
    if (empty($password)) {
        $errors[] = 'رمز عبور نمی‌تواند خالی باشد';
    } elseif (strlen($password) < 1) {
        $errors[] = 'رمز عبور نامعتبر است';
    }
    
    return $errors;
}

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $error = 'توکن امنیتی نامعتبر است';
        logSecurityEvent('login_csrf_attempt', 'CSRF token mismatch', 'high');
    } else {
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        
        // Enhanced input validation
        $validationErrors = validateLoginInput($username, $password);
        if (!empty($validationErrors)) {
            $error = implode('<br>', $validationErrors);
            logSecurityEvent('login_validation_failed', 'Username: ' . $username . ' - Errors: ' . implode(', ', $validationErrors), 'medium');
        } else {
            // Security checks
            if (isIPBlacklisted(getClientIP())) {
                $error = 'دسترسی شما مسدود شده است';
                logSecurityEvent('login_blocked_blacklisted_ip', 'IP: ' . getClientIP(), 'high');
            } elseif (!checkRateLimit('login', 5)) {
                $error = 'تعداد تلاش‌های ورود بیش از حد مجاز است. لطفاً کمی صبر کنید';
                logSecurityEvent('login_rate_limit_exceeded', 'IP: ' . getClientIP(), 'medium');
            } else {
                try {
                    $db = new Database();
                    $conn = $db->getConnection();
                    
                    // Check for XSS or SQL injection (only on username, not password)
                    if (detectXSS($username) || detectSQLInjection($username)) {
                        $error = 'نام کاربری نامعتبر';
                        logSecurityEvent('login_xss_sql_attempt', 'Username: ' . $username, 'high');
                    } else {
                        // Get user from database
                        $query = "SELECT id, username, password, is_active, email FROM admin_users WHERE username = ?";
                        $stmt = $conn->prepare($query);
                        $stmt->execute([$username]);
                        $user = $stmt->fetch();
                        
                        if ($user && $user['is_active'] && password_verify($password, $user['password'])) {
                            // Check if 2FA is enabled in system settings
                            $twofaEnabled = getSystemSetting('twofa_enabled');
                            
                            if ($twofaEnabled == '1' && !empty($user['email'])) {
                                // 2FA is enabled for all users with email, generate and send verification code
                                $code = sprintf('%06d', mt_rand(100000, 999999));
                                $expirySeconds = (int)getSystemSetting('twofa_code_expiry') ?: 300; // Default 5 minutes
                                $expiryMinutes = $expirySeconds / 60;
                                
                                // Use MySQL NOW() to avoid timezone issues
                                $insertQuery = "INSERT INTO verification_codes (user_id, code, email, expires_at) 
                                               VALUES (?, ?, ?, DATE_ADD(NOW(), INTERVAL ? SECOND))";
                                $insertStmt = $conn->prepare($insertQuery);
                                $insertStmt->execute([$user['id'], $code, $user['email'], $expirySeconds]);
                                
                                // Send email with code
                                require_once 'config/email-config.php';
                                $subject = 'کد تایید ورود - پنل مدیریت گواهیتو';
                                $message = "
                                <h3 style='color: #041c3f; margin-bottom: 20px;'>🔐 کد تایید ورود</h3>
                                
                                <div style='background: #f8f9fa; border: 1px solid #e9ecef; border-radius: 8px; padding: 20px; margin: 20px 0; text-align: center;'>
                                    <h2 style='color: #333; font-size: 2em; margin: 0; letter-spacing: 5px;'>$code</h2>
                                </div>
                                
                                <div style='background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                                    <p style='margin: 0; color: #856404;'><strong>⚠️ توجه:</strong> این کد تا $expiryMinutes دقیقه معتبر است</p>
                                </div>
                                
                                <div style='background: rgba(4, 28, 63, 0.1); border: 1px solid #041c3f; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                                    <p style='margin: 0; color: #041c3f;'>🕐 تاریخ ارسال: " . date('Y/m/d H:i:s') . "</p>
                                </div>";
                                
                                // Log email attempt
                                error_log("Attempting to send 2FA email to: " . $user['email']);
                                
                                // Try to send email, but don't fail if it doesn't work
                                $emailResult = @sendEmail($user['email'], $subject, $message);
                                error_log("Email send result: " . ($emailResult ? 'SUCCESS' : 'FAILED'));
                                
                                // Set 2FA session variables regardless of email result
                                $_SESSION['2fa_user_id'] = $user['id'];
                                $_SESSION['2fa_username'] = $user['username'];
                                $_SESSION['2fa_email'] = $user['email'];
                                $_SESSION['2fa_expires'] = time() + ($expiryMinutes * 60);
                                
                                if ($emailResult) {
                                    logSecurityEvent('2fa_code_sent', 'User: ' . $username . ' Email: ' . $user['email'], 'low');
                                } else {
                                    logSecurityEvent('2fa_code_sent_no_email', 'User: ' . $username . ' Email: ' . $user['email'], 'medium');
                                }
                                
                                error_log("Redirecting to verify-2fa.php");
                                header('Location: verify-2fa.php');
                                exit();
                            } else {
                                // No email configured, proceed with normal login
                                $_SESSION['admin_id'] = $user['id'];
                                $_SESSION['admin_username'] = $user['username'];
                                $_SESSION['admin_role'] = 'admin';
                                $_SESSION['last_activity'] = time();
                                
                                // Update last login
                                $updateQuery = "UPDATE admin_users SET last_login = NOW() WHERE id = ?";
                                $updateStmt = $conn->prepare($updateQuery);
                                $updateStmt->execute([$user['id']]);
                                
                                logSecurityEvent('login_successful_hidden', 'User: ' . $username, 'low');
                                
                                header('Location: dashboard.php');
                                exit();
                            }
                        } else {
                            $error = 'نام کاربری یا رمز عبور اشتباه است';
                            logSecurityEvent('login_failed_hidden', 'Username: ' . $username, 'medium');
                            
                            // Check for multiple failed login attempts
                            $failedAttempts = getFailedLoginAttempts($clientIP);
                            $maxAttempts = (int)getSystemSetting('max_login_attempts') ?: 5;
                            $lockoutDuration = (int)getSystemSetting('lockout_duration_minutes') ?: 15;
                            
                            if ($failedAttempts >= $maxAttempts) {
                                // Auto-block IP for configured duration
                                autoBlockIP($clientIP, "$maxAttempts بار تلاش ناموفق برای ورود", $lockoutDuration);
                                $error = "به دلیل تلاش‌های مکرر ناموفق، IP شما برای $lockoutDuration دقیقه مسدود شده است";
                                logSecurityEvent('ip_auto_blocked_login_failures', "IP auto-blocked due to $maxAttempts failed login attempts", 'high', json_encode([
                                    'ip' => $clientIP,
                                    'failed_attempts' => $failedAttempts,
                                    'max_attempts' => $maxAttempts,
                                    'lockout_duration' => $lockoutDuration
                                ]));
                            }
                        }
                    }
                } catch (Exception $e) {
                    error_log("Login error: " . $e->getMessage());
                    $error = 'خطا در سیستم. لطفاً دوباره تلاش کنید';
                }
            }
        }
    }
}

// Check for timeout message
if (isset($_GET['timeout'])) {
    $error = 'جلسه شما منقضی شده است. لطفاً دوباره وارد شوید';
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ورود به پنل مدیریت - گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
</head>
<body>
    <div class="main-content" style="max-width: 500px; margin: 2rem auto; padding: 0 1rem;">
        <div class="page-header text-center">
            <h1 class="page-title">🔐 ورود به پنل مدیریت</h1>
            <p class="page-description">گواهیتو Admin Panel - پنل ادمین گواهیتو</p>
        </div>
        
        <div class="card fade-in" style="box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);">
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <div style="display: flex; align-items: center; gap: 10px;">
                        <span style="font-size: 1.2em;">⚠️</span>
                        <span><?php echo $error; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success">
                    <div style="display: flex; align-items: center; gap: 10px;">
                        <span style="font-size: 1.2em;">✅</span>
                        <span><?php echo htmlspecialchars($success); ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="" id="loginForm">
                <!-- CSRF Token -->
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                
                <div class="form-group">
                    <label for="username" class="form-label">
                        <span style="margin-left: 8px;">👤</span>
                        نام کاربری
                    </label>
                    <input type="text" id="username" name="username" class="form-input" 
                           value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>" 
                           required autocomplete="username" 
                           placeholder="نام کاربری خود را وارد کنید"
                           pattern="[a-zA-Z0-9_]{3,50}"
                           title="نام کاربری باید 3 تا 50 کاراکتر و فقط شامل حروف، اعداد و خط زیر باشد">
                </div>
                
                <div class="form-group">
                    <label for="password" class="form-label">
                        <span style="margin-left: 8px;">🔒</span>
                        رمز عبور
                    </label>
                    <div style="position: relative;">
                        <input type="password" id="password" name="password" class="form-input" 
                               required autocomplete="current-password"
                               placeholder="رمز عبور خود را وارد کنید"
                               minlength="1">
                        <button type="button" id="togglePassword" style="position: absolute; left: 10px; top: 50%; transform: translateY(-50%); background: none; border: none; color: var(--dark-400); cursor: pointer; font-size: 1.1em;">
                            👁️
                        </button>
                    </div>
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn btn-primary btn-lg" style="width: 100%; padding: 15px; font-size: 1.1em; font-weight: 600;">
                        <span style="margin-left: 8px;">🚀</span>
                        ورود به سیستم
                    </button>
                </div>
            </form>
            
            <div style="text-align: center; margin-top: 2rem; padding-top: 1rem; border-top: 1px solid var(--dark-600);">
                <div style="background: var(--dark-700); padding: 15px; border-radius: 8px; margin-bottom: 15px;">
                    <p style="color: var(--dark-300); font-size: 0.9rem; margin: 0;">
                        <span style="color: var(--primary-400); font-weight: 600;">🔐 امنیت دو مرحله‌ای:</span>
                        <br>بعد از ورود، کد تایید به ایمیل شما ارسال خواهد شد
                    </p>
                </div>
                <div style="background: var(--dark-700); padding: 15px; border-radius: 8px; margin-bottom: 15px;">
                    <p style="color: var(--dark-300); font-size: 0.9rem; margin: 0;">
                        <span style="color: var(--primary-400); font-weight: 600;">🛡️ سیستم امنیتی پیشرفته:</span>
                        <br>• تمامی فعالیت‌ها ثبت و نظارت می‌شود<br>
                    
                    </p>
                </div>
                <p style="color: var(--dark-400); font-size: 0.875rem;">
                    برای ورود به سیستم، نام کاربری و رمز عبور خود را وارد کنید
                </p>
            </div>
        </div>
        
        <div style="text-align: center; margin-top: 2rem;">
            <p style="color: var(--dark-400); font-size: 0.875rem;">
                © 2024 گواهیتو Admin Panel - <a href="https://gavahito.ir/" style="color: var(--primary-400); text-decoration: none;">gavahito.ir</a>
            </p>
        </div>
    </div>
    
    <script>
        // Auto-focus on username field
        document.getElementById('username').focus();
        
        // Toggle password visibility
        document.getElementById('togglePassword').addEventListener('click', function() {
            const passwordInput = document.getElementById('password');
            const toggleButton = this;
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                toggleButton.textContent = '🙈';
                toggleButton.title = 'مخفی کردن رمز عبور';
            } else {
                passwordInput.type = 'password';
                toggleButton.textContent = '👁️';
                toggleButton.title = 'نمایش رمز عبور';
            }
        });
        
        // Enhanced form validation
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            const username = document.getElementById('username').value.trim();
            const password = document.getElementById('password').value;
            const submitButton = this.querySelector('button[type="submit"]');
            
            // Clear previous errors
            clearValidationErrors();
            
            let hasErrors = false;
            
            // Username validation
            if (!username) {
                showFieldError('username', 'نام کاربری نمی‌تواند خالی باشد');
                hasErrors = true;
            } else if (username.length < 3) {
                showFieldError('username', 'نام کاربری باید حداقل 3 کاراکتر باشد');
                hasErrors = true;
            } else if (username.length > 50) {
                showFieldError('username', 'نام کاربری نمی‌تواند بیش از 50 کاراکتر باشد');
                hasErrors = true;
            } else if (!/^[a-zA-Z0-9_]+$/.test(username)) {
                showFieldError('username', 'نام کاربری فقط می‌تواند شامل حروف، اعداد و خط زیر باشد');
                hasErrors = true;
            }
            
            // Password validation
            if (!password) {
                showFieldError('password', 'رمز عبور نمی‌تواند خالی باشد');
                hasErrors = true;
            }
            
            if (hasErrors) {
                e.preventDefault();
                showNotification('لطفاً خطاهای فرم را برطرف کنید', 'error');
                return false;
            }
            
            // Show loading state
            submitButton.disabled = true;
            submitButton.innerHTML = '<span style="margin-left: 8px;">⏳</span>در حال بررسی...';
        });
        
        // Show field error
        function showFieldError(fieldId, message) {
            const field = document.getElementById(fieldId);
            const errorDiv = document.createElement('div');
            errorDiv.className = 'field-error';
            errorDiv.style.cssText = 'color: #ef4444; font-size: 0.875rem; margin-top: 5px; display: flex; align-items: center; gap: 5px;';
            errorDiv.innerHTML = '<span>⚠️</span>' + message;
            
            field.parentNode.appendChild(errorDiv);
            field.style.borderColor = '#ef4444';
        }
        
        // Clear validation errors
        function clearValidationErrors() {
            const errors = document.querySelectorAll('.field-error');
            errors.forEach(error => error.remove());
            
            const inputs = document.querySelectorAll('.form-input');
            inputs.forEach(input => {
                input.style.borderColor = '';
            });
        }
        
        // Show notification function
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 15px 20px;
                border-radius: 8px;
                color: white;
                font-weight: 600;
                z-index: 10000;
                animation: slideIn 0.3s ease-out;
                max-width: 300px;
            `;
            
            if (type === 'error') {
                notification.style.background = '#ef4444';
            } else if (type === 'success') {
                notification.style.background = '#10b981';
            } else {
                notification.style.background = '#3b82f6';
            }
            
            notification.textContent = message;
            document.body.appendChild(notification);
            
            // Remove after 3 seconds
            setTimeout(() => {
                notification.style.animation = 'slideOut 0.3s ease-in';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }
        
        // Add CSS animations
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideIn {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOut {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
            
            @media (max-width: 768px) {
                .main-content {
                    margin: 1rem auto !important;
                    padding: 0 0.5rem !important;
                }
                .card {
                    padding: 1.5rem !important;
                }
                .form-input {
                    font-size: 16px !important; /* Prevent zoom on iOS */
                }
            }
        `;
        document.head.appendChild(style);
        
        // Check if page is loaded after form submission (back button or refresh)
        window.addEventListener('load', function() {
            const submitButton = document.querySelector('button[type="submit"]');
            if (submitButton && submitButton.disabled) {
                // Reset button if it's still in loading state
                submitButton.disabled = false;
                submitButton.innerHTML = '<span style="margin-left: 8px;">🚀</span>ورود به سیستم';
            }
        });
    </script>
</body>
</html>
