<?php
/**
 * IP Geolocation System - سیستم تشخیص کشور IP
 */

function getIPCountry($ip) {
    // Skip local IPs
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
        return 'Local';
    }
    
    try {
        // Use ipapi.co service (free, no API key required)
        $url = "http://ip-api.com/json/{$ip}?fields=status,message,country,countryCode,region,regionName,city,timezone,isp,org,as,mobile,proxy,hosting";
        
        $context = stream_context_create([
            'http' => [
                'timeout' => 3,
                'user_agent' => 'Gavahito-IP-Check/1.0'
            ]
        ]);
        
        $response = @file_get_contents($url, false, $context);
        
        if ($response === false) {
            return 'Unknown';
        }
        
        $data = json_decode($response, true);
        
        if ($data && isset($data['status']) && $data['status'] === 'success') {
            $country = $data['country'] ?? 'Unknown';
            $city = $data['city'] ?? '';
            $isp = $data['isp'] ?? '';
            
            $result = $country;
            if ($city) {
                $result .= " ({$city})";
            }
            if ($isp) {
                $result .= " - {$isp}";
            }
            
            return $result;
        }
        
        return 'Unknown';
        
    } catch (Exception $e) {
        error_log("IP geolocation error: " . $e->getMessage());
        return 'Unknown';
    }
}

function getIPDetails($ip) {
    // Skip local IPs
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
        return [
            'country' => 'Local',
            'city' => 'Local Network',
            'isp' => 'Local',
            'proxy' => false,
            'hosting' => false,
            'mobile' => false
        ];
    }
    
    try {
        $url = "http://ip-api.com/json/{$ip}?fields=status,message,country,countryCode,region,regionName,city,timezone,isp,org,as,mobile,proxy,hosting";
        
        $context = stream_context_create([
            'http' => [
                'timeout' => 3,
                'user_agent' => 'Gavahito-IP-Check/1.0'
            ]
        ]);
        
        $response = @file_get_contents($url, false, $context);
        
        if ($response === false) {
            return [
                'country' => 'Unknown',
                'city' => 'Unknown',
                'isp' => 'Unknown',
                'proxy' => false,
                'hosting' => false,
                'mobile' => false
            ];
        }
        
        $data = json_decode($response, true);
        
        if ($data && isset($data['status']) && $data['status'] === 'success') {
            return [
                'country' => $data['country'] ?? 'Unknown',
                'city' => $data['city'] ?? 'Unknown',
                'isp' => $data['isp'] ?? 'Unknown',
                'proxy' => $data['proxy'] ?? false,
                'hosting' => $data['hosting'] ?? false,
                'mobile' => $data['mobile'] ?? false
            ];
        }
        
        return [
            'country' => 'Unknown',
            'city' => 'Unknown',
            'isp' => 'Unknown',
            'proxy' => false,
            'hosting' => false,
            'mobile' => false
        ];
        
    } catch (Exception $e) {
        error_log("IP details error: " . $e->getMessage());
        return [
            'country' => 'Unknown',
            'city' => 'Unknown',
            'isp' => 'Unknown',
            'proxy' => false,
            'hosting' => false,
            'mobile' => false
        ];
    }
}

function getIPGeolocation($ip) {
    // Skip local IPs
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
        return [
            'country' => 'Local',
            'city' => 'Local Network',
            'isp' => 'Local',
            'proxy' => false,
            'hosting' => false,
            'mobile' => false
        ];
    }
    
    try {
        $url = "http://ip-api.com/json/{$ip}?fields=status,message,country,countryCode,region,regionName,city,timezone,isp,org,as,mobile,proxy,hosting";
        
        $context = stream_context_create([
            'http' => [
                'timeout' => 3,
                'user_agent' => 'Gavahito-IP-Check/1.0'
            ]
        ]);
        
        $response = @file_get_contents($url, false, $context);
        
        if ($response === false) {
            return [
                'country' => 'Unknown',
                'city' => 'Unknown',
                'isp' => 'Unknown',
                'proxy' => false,
                'hosting' => false,
                'mobile' => false
            ];
        }
        
        $data = json_decode($response, true);
        
        if ($data && isset($data['status']) && $data['status'] === 'success') {
            return [
                'country' => $data['country'] ?? 'Unknown',
                'city' => $data['city'] ?? 'Unknown',
                'isp' => $data['isp'] ?? 'Unknown',
                'proxy' => $data['proxy'] ?? false,
                'hosting' => $data['hosting'] ?? false,
                'mobile' => $data['mobile'] ?? false
            ];
        }
        
        return [
            'country' => 'Unknown',
            'city' => 'Unknown',
            'isp' => 'Unknown',
            'proxy' => false,
            'hosting' => false,
            'mobile' => false
        ];
        
    } catch (Exception $e) {
        error_log("IP geolocation error: " . $e->getMessage());
        return [
            'country' => 'Unknown',
            'city' => 'Unknown',
            'isp' => 'Unknown',
            'proxy' => false,
            'hosting' => false,
            'mobile' => false
        ];
    }
}

// Cache IP geolocation results to avoid repeated API calls
function getCachedIPCountry($ip) {
    $cacheFile = __DIR__ . '/../cache/ip_cache.json';
    $cacheDir = dirname($cacheFile);
    
    // Create cache directory if it doesn't exist
    if (!is_dir($cacheDir)) {
        mkdir($cacheDir, 0755, true);
    }
    
    // Load cache
    $cache = [];
    if (file_exists($cacheFile)) {
        $cacheData = file_get_contents($cacheFile);
        $cache = json_decode($cacheData, true) ?: [];
    }
    
    // Check if we have a recent cache entry (less than 24 hours old)
    if (isset($cache[$ip]) && isset($cache[$ip]['timestamp'])) {
        $age = time() - $cache[$ip]['timestamp'];
        if ($age < 86400) { // 24 hours
            return $cache[$ip]['country'];
        }
    }
    
    // Get fresh data
    $country = getIPCountry($ip);
    
    // Cache the result
    $cache[$ip] = [
        'country' => $country,
        'timestamp' => time()
    ];
    
    // Save cache (keep only last 1000 entries)
    if (count($cache) > 1000) {
        $cache = array_slice($cache, -1000, 1000, true);
    }
    
    file_put_contents($cacheFile, json_encode($cache));
    
    return $country;
}
?>
