<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

require_once 'config/database.php';
require_once 'includes/functions.php';
require_once 'includes/ip-geolocation.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to ip-management.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$db = new Database();
$conn = $db->getConnection();



// Handle IP actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $ip = sanitizeInput($_POST['ip'] ?? '');
    $reason = sanitizeInput($_POST['reason'] ?? '');
    $duration = (int)($_POST['duration'] ?? 24); // hours
    
    if ($action === 'block' && $ip) {
        try {
            $expiresAt = $duration > 0 ? date('Y-m-d H:i:s', strtotime("+$duration hours")) : null;
            
            // Get valid admin_id for added_by field
            $adminId = $_SESSION['admin_id'] ?? 1;
            
            // Verify admin exists, if not use default admin (id=1)
            $adminStmt = $conn->prepare("SELECT id FROM admin_users WHERE id = ?");
            $adminStmt->execute([$adminId]);
            $adminExists = $adminStmt->fetch();
            
            if (!$adminExists) {
                // Use default admin (id=1) or first available admin
                $adminStmt = $conn->prepare("SELECT id FROM admin_users LIMIT 1");
                $adminStmt->execute();
                $firstAdmin = $adminStmt->fetch();
                $adminId = $firstAdmin['id'] ?? 1;
            }
            
            $query = "INSERT INTO ip_blacklist (ip_address, reason, expires_at, added_by, created_at) VALUES (?, ?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE reason = ?, expires_at = ?, updated_at = NOW()";
            $stmt = $conn->prepare($query);
            $stmt->execute([$ip, $reason, $expiresAt, $adminId, $reason, $expiresAt]);
            
            logSecurityEvent('ip_blocked', 'IP blocked by admin', 'medium', json_encode(['blocked_ip' => $ip, 'reason' => $reason, 'duration' => $duration]));
                
            header('Location: ip-management.php?success=blocked');
            exit();
        } catch (Exception $e) {
            $error = 'خطا در مسدود کردن IP: ' . $e->getMessage();
        }
    } elseif ($action === 'unblock' && $ip) {
        try {
            $query = "DELETE FROM ip_blacklist WHERE ip_address = ?";
            $stmt = $conn->prepare($query);
            $stmt->execute([$ip]);
            
            logSecurityEvent('ip_unblocked', 'IP unblocked by admin', 'low', json_encode(['unblocked_ip' => $ip]));
                
            header('Location: ip-management.php?success=unblocked');
            exit();
        } catch (Exception $e) {
            $error = 'خطا در آزادسازی IP: ' . $e->getMessage();
        }
    } elseif ($action === 'whitelist' && $ip) {
        try {
            // Get valid admin_id for added_by field
            $adminId = $_SESSION['admin_id'] ?? 1;
            
            // Verify admin exists, if not use default admin (id=1)
            $adminStmt = $conn->prepare("SELECT id FROM admin_users WHERE id = ?");
            $adminStmt->execute([$adminId]);
            $adminExists = $adminStmt->fetch();
            
            if (!$adminExists) {
                // Use default admin (id=1) or first available admin
                $adminStmt = $conn->prepare("SELECT id FROM admin_users LIMIT 1");
                $adminStmt->execute();
                $firstAdmin = $adminStmt->fetch();
                $adminId = $firstAdmin['id'] ?? 1;
            }
            
            // Check if updated_at column exists in ip_whitelist
            $checkColumn = $conn->query("SHOW COLUMNS FROM ip_whitelist LIKE 'updated_at'");
            if ($checkColumn->rowCount() > 0) {
                $query = "INSERT INTO ip_whitelist (ip_address, description, added_by, created_at) VALUES (?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE description = ?, updated_at = NOW()";
                $stmt = $conn->prepare($query);
                $stmt->execute([$ip, $reason, $adminId, $reason]);
            } else {
                $query = "INSERT INTO ip_whitelist (ip_address, description, added_by, created_at) VALUES (?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE description = ?";
                $stmt = $conn->prepare($query);
                $stmt->execute([$ip, $reason, $adminId, $reason]);
            }
            
            logSecurityEvent('ip_whitelisted', 'IP whitelisted by admin', 'low', json_encode(['whitelisted_ip' => $ip, 'reason' => $reason]));
                
            header('Location: ip-management.php?success=whitelisted');
            exit();
        } catch (Exception $e) {
            $error = 'خطا در اضافه کردن IP به لیست سفید: ' . $e->getMessage();
        }
    } elseif ($action === 'remove_whitelist' && $ip) {
        try {
            $query = "DELETE FROM ip_whitelist WHERE ip_address = ?";
            $stmt = $conn->prepare($query);
            $stmt->execute([$ip]);
            
            logSecurityEvent('ip_whitelist_removed', 'IP removed from whitelist by admin', 'low', json_encode(['removed_ip' => $ip]));
                
            header('Location: ip-management.php?success=whitelist_removed');
            exit();
        } catch (Exception $e) {
            $error = 'خطا در حذف IP از لیست سفید: ' . $e->getMessage();
        }
    }
}

// Get blacklisted IPs
$blacklistQuery = "SELECT * FROM ip_blacklist ORDER BY created_at DESC";
$blacklistStmt = $conn->query($blacklistQuery);
$blacklistedIPs = $blacklistStmt->fetchAll();

// Get whitelisted IPs
$whitelistQuery = "SELECT * FROM ip_whitelist ORDER BY created_at DESC";
$whitelistStmt = $conn->query($whitelistQuery);
$whitelistedIPs = $whitelistStmt->fetchAll();

// Get statistics
$totalBlocked = count($blacklistedIPs);
$totalWhitelisted = count($whitelistedIPs);
$expiredBlocks = 0;
$activeBlocks = 0;

foreach ($blacklistedIPs as $ip) {
    if ($ip['expires_at'] && strtotime($ip['expires_at']) < time()) {
        $expiredBlocks++;
    } else {
        $activeBlocks++;
    }
}

// Get suspicious IPs (IPs with high request rates)
$suspiciousIPsQuery = "
    SELECT 
        ip_address,
        COUNT(*) as request_count,
        MAX(created_at) as last_request,
        MIN(created_at) as first_request,
        GROUP_CONCAT(DISTINCT event_type ORDER BY event_type SEPARATOR ', ') as event_types
    FROM security_logs 
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
    GROUP BY ip_address 
    HAVING request_count >= 10
    ORDER BY request_count DESC
    LIMIT 50
";
$suspiciousIPsStmt = $conn->query($suspiciousIPsQuery);
$suspiciousIPs = $suspiciousIPsStmt->fetchAll();

// Get recent security events
$recentEventsQuery = "
    SELECT 
        ip_address,
        event_type,
        severity,
        created_at,
        details
    FROM security_logs 
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
    AND severity IN ('high', 'critical')
    ORDER BY created_at DESC
    LIMIT 20
";
$recentEventsStmt = $conn->query($recentEventsQuery);
$recentEvents = $recentEventsStmt->fetchAll();


?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت IP ها - گواهیتو</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/admin-style.css">
    <?php include 'includes/admin-styles.php'; ?>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <!-- Success Messages -->
        <?php if (isset($_GET['success'])): ?>
            <div class="alert alert-success mb-6">
                <?php
                switch ($_GET['success']) {
                    case 'blocked':
                        echo 'IP با موفقیت مسدود شد';
                        break;
                    case 'unblocked':
                        echo 'IP با موفقیت آزاد شد';
                        break;
                    case 'whitelisted':
                        echo 'IP با موفقیت به لیست سفید اضافه شد';
                        break;
                    case 'whitelist_removed':
                        echo 'IP با موفقیت از لیست سفید حذف شد';
                        break;
                }
                ?>
            </div>
        <?php endif; ?>

        <!-- Error Messages -->
        <?php if (isset($error)): ?>
            <div class="alert alert-error mb-6">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>





        <!-- Statistics Cards -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
            <div class="card">
                <div class="flex items-center">
                    <div class="p-3 rounded-full bg-red-500/10 text-red-500">
                        <i class="fas fa-ban text-xl"></i>
                    </div>
                    <div class="mr-4">
                        <p class="text-sm text-gray-400">IP های مسدود شده</p>
                        <p class="text-2xl font-bold text-white"><?php echo $totalBlocked; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="flex items-center">
                    <div class="p-3 rounded-full bg-green-500/10 text-green-500">
                        <i class="fas fa-check-circle text-xl"></i>
                    </div>
                    <div class="mr-4">
                        <p class="text-sm text-gray-400">IP های سفید</p>
                        <p class="text-2xl font-bold text-white"><?php echo $totalWhitelisted; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="flex items-center">
                    <div class="p-3 rounded-full bg-yellow-500/10 text-yellow-500">
                        <i class="fas fa-clock text-xl"></i>
                    </div>
                    <div class="mr-4">
                        <p class="text-sm text-gray-400">مسدودیت فعال</p>
                        <p class="text-2xl font-bold text-white"><?php echo $activeBlocks; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="flex items-center">
                    <div class="p-3 rounded-full bg-gray-500/10 text-gray-500">
                        <i class="fas fa-hourglass-end text-xl"></i>
                    </div>
                    <div class="mr-4">
                        <p class="text-sm text-gray-400">منقضی شده</p>
                        <p class="text-2xl font-bold text-white"><?php echo $expiredBlocks; ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- IP Management Forms -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
            <!-- Block IP Form -->
            <div class="card">
                <h3 class="text-lg font-semibold text-white mb-4">
                    <i class="fas fa-ban text-red-500 ml-2"></i>
                    مسدود کردن IP
                </h3>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="action" value="block">
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">آدرس IP</label>
                        <input type="text" name="ip" required 
                               class="form-input" 
                               placeholder="192.168.1.1">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">دلیل</label>
                        <textarea name="reason" rows="3" 
                                  class="form-textarea" 
                                  placeholder="دلیل مسدود کردن IP"></textarea>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">مدت زمان (ساعت)</label>
                        <input type="number" name="duration" value="24" min="1" max="8760" 
                               class="form-input">
                        <p class="text-xs text-gray-400 mt-1">0 = دائمی</p>
                    </div>
                    
                    <button type="submit" class="btn btn-danger w-full">
                        <i class="fas fa-ban ml-2"></i>
                        مسدود کردن
                    </button>
                </form>
            </div>

            <!-- Whitelist IP Form -->
            <div class="card">
                <h3 class="text-lg font-semibold text-white mb-4">
                    <i class="fas fa-check-circle text-green-500 ml-2"></i>
                    اضافه کردن به لیست سفید
                </h3>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="action" value="whitelist">
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">آدرس IP</label>
                        <input type="text" name="ip" required 
                               class="form-input" 
                               placeholder="192.168.1.1">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">توضیحات</label>
                        <textarea name="reason" rows="3" 
                                  class="form-textarea" 
                                  placeholder="توضیحات IP"></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-success w-full">
                        <i class="fas fa-check-circle ml-2"></i>
                        اضافه کردن
                    </button>
                </form>
            </div>
        </div>

        <!-- Suspicious IPs Section -->
        <div class="card mb-8">
            <h3 class="text-lg font-semibold text-white mb-4">
                <i class="fas fa-exclamation-triangle text-yellow-500 ml-2"></i>
                IP های مشکوک (24 ساعت گذشته)
            </h3>
            
            <?php if (empty($suspiciousIPs)): ?>
                <p class="text-gray-400 text-center py-8">هیچ IP مشکوکی در 24 ساعت گذشته یافت نشد</p>
            <?php else: ?>
                <div class="overflow-x-auto">
                    <table class="table">
                                                    <thead>
                                <tr>
                                    <th>IP</th>
                                    <th>کشور</th>
                                    <th>تعداد درخواست</th>
                                    <th>نوع رویداد</th>
                                    <th>آخرین درخواست</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                        <tbody>
                            <?php foreach ($suspiciousIPs as $ip): ?>
                                <?php 
                                $isBlacklisted = false;
                                $isWhitelisted = false;
                                foreach ($blacklistedIPs as $blacklisted) {
                                    if ($blacklisted['ip_address'] === $ip['ip_address']) {
                                        $isBlacklisted = true;
                                        break;
                                    }
                                }
                                foreach ($whitelistedIPs as $whitelisted) {
                                    if ($whitelisted['ip_address'] === $ip['ip_address']) {
                                        $isWhitelisted = true;
                                        break;
                                    }
                                }
                                ?>
                                <tr class="<?php echo $isBlacklisted ? 'bg-red-900/20' : ($isWhitelisted ? 'bg-green-900/20' : ''); ?>">
                                    <td>
                                        <code class="text-sm"><?php echo htmlspecialchars($ip['ip_address']); ?></code>
                                        <?php if ($isBlacklisted): ?>
                                            <span class="badge badge-danger text-xs">مسدود شده</span>
                                        <?php elseif ($isWhitelisted): ?>
                                            <span class="badge badge-success text-xs">لیست سفید</span>
                                        <?php endif; ?>

                                    </td>
                                    <td>
                                        <span class="text-xs text-gray-300">
                                            <?php echo getCachedIPCountry($ip['ip_address']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="font-bold <?php echo $ip['request_count'] > 50 ? 'text-red-400' : ($ip['request_count'] > 20 ? 'text-yellow-400' : 'text-green-400'); ?>">
                                            <?php echo $ip['request_count']; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="text-xs text-gray-400">
                                            <?php echo htmlspecialchars($ip['event_types']); ?>
                                        </span>
                                    </td>
                                    <td>

                                    </td>
                                    <td>
                                        <span class="text-xs text-gray-400">
                                            <?php echo date('H:i', strtotime($ip['last_request'])); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if (!$isBlacklisted && !$isWhitelisted): ?>
                                            <div class="flex gap-2">
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="action" value="block">
                                                    <input type="hidden" name="ip" value="<?php echo htmlspecialchars($ip['ip_address']); ?>">
                                                    <input type="hidden" name="reason" value="IP مشکوک - <?php echo $ip['request_count']; ?> درخواست در 24 ساعت">
                                                    <input type="hidden" name="duration" value="24">
                                                    <button type="submit" class="btn btn-sm btn-danger" 
                                                            onclick="return confirm('آیا از مسدود کردن این IP اطمینان دارید؟')">
                                                        <i class="fas fa-ban"></i> مسدود
                                                    </button>
                                                </form>
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="action" value="whitelist">
                                                    <input type="hidden" name="ip" value="<?php echo htmlspecialchars($ip['ip_address']); ?>">
                                                    <input type="hidden" name="reason" value="IP معتبر - اضافه شده از بخش مشکوک">
                                                    <button type="submit" class="btn btn-sm btn-success" 
                                                            onclick="return confirm('آیا از اضافه کردن این IP به لیست سفید اطمینان دارید؟')">
                                                        <i class="fas fa-check"></i> سفید
                                                    </button>
                                                </form>
                                            </div>
                                        <?php elseif ($isBlacklisted): ?>
                                            <span class="text-red-400 text-sm">مسدود شده</span>
                                        <?php elseif ($isWhitelisted): ?>
                                            <span class="text-green-400 text-sm">لیست سفید</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>

        <!-- Recent Security Events -->
        <div class="card mb-8">
            <h3 class="text-lg font-semibold text-white mb-4">
                <i class="fas fa-shield-alt text-blue-500 ml-2"></i>
                رویدادهای امنیتی اخیر (24 ساعت گذشته)
            </h3>
            
            <?php if (empty($recentEvents)): ?>
                <p class="text-gray-400 text-center py-8">هیچ رویداد امنیتی مهمی در 24 ساعت گذشته یافت نشد</p>
            <?php else: ?>
                <div class="space-y-3">
                    <?php foreach ($recentEvents as $event): ?>
                        <div class="border border-gray-700 rounded-lg p-3 <?php echo $event['severity'] === 'critical' ? 'bg-red-900/20 border-red-500' : 'bg-yellow-900/20 border-yellow-500'; ?>">
                            <div class="flex justify-between items-start">
                                <div class="flex-1">
                                    <div class="flex items-center gap-2 mb-1">
                                        <code class="text-sm text-gray-300"><?php echo htmlspecialchars($event['ip_address']); ?></code>
                                        <span class="badge <?php echo $event['severity'] === 'critical' ? 'badge-danger' : 'badge-warning'; ?> text-xs">
                                            <?php echo $event['severity'] === 'critical' ? 'بحرانی' : 'بالا'; ?>
                                        </span>
                                    </div>
                                    <div class="text-sm text-gray-400">
                                        <strong><?php echo htmlspecialchars($event['event_type']); ?></strong>
                                        <?php if ($event['details']): ?>
                                            - <?php echo htmlspecialchars(substr($event['details'], 0, 100)); ?>
                                            <?php if (strlen($event['details']) > 100): ?>...<?php endif; ?>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="text-xs text-gray-500">
                                    <?php echo date('H:i', strtotime($event['created_at'])); ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- IP Lists -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <!-- Blacklisted IPs -->
            <div class="card">
                <h3 class="text-lg font-semibold text-white mb-4">
                    <i class="fas fa-ban text-red-500 ml-2"></i>
                    IP های مسدود شده
                </h3>
                
                <?php if (empty($blacklistedIPs)): ?>
                    <p class="text-gray-400 text-center py-8">هیچ IP مسدود شده‌ای وجود ندارد</p>
                <?php else: ?>
                    <div class="overflow-x-auto">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>IP</th>
                                    <th>دلیل</th>
                                    <th>تاریخ انقضا</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($blacklistedIPs as $ip): ?>
                                    <tr>
                                        <td>
                                            <code class="text-sm"><?php echo htmlspecialchars($ip['ip_address']); ?></code>
                                        </td>
                                        <td><?php echo htmlspecialchars($ip['reason']); ?></td>
                                        <td>
                                            <?php if ($ip['expires_at']): ?>
                                                <?php 
                                                $expires = strtotime($ip['expires_at']);
                                                $isExpired = $expires < time();
                                                $class = $isExpired ? 'text-red-400' : 'text-green-400';
                                                ?>
                                                <span class="<?php echo $class; ?>">
                                                    <?php echo date('Y/m/d H:i', $expires); ?>
                                                    <?php if ($isExpired): ?>
                                                        <span class="text-xs">(منقضی شده)</span>
                                                    <?php endif; ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="text-yellow-400">دائمی</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <form method="POST" class="inline">
                                                <input type="hidden" name="action" value="unblock">
                                                <input type="hidden" name="ip" value="<?php echo htmlspecialchars($ip['ip_address']); ?>">
                                                <button type="submit" class="btn btn-sm btn-success" 
                                                        onclick="return confirm('آیا از آزادسازی این IP اطمینان دارید؟')">
                                                    <i class="fas fa-unlock"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Whitelisted IPs -->
            <div class="card">
                <h3 class="text-lg font-semibold text-white mb-4">
                    <i class="fas fa-check-circle text-green-500 ml-2"></i>
                    IP های لیست سفید
                </h3>
                
                <?php if (empty($whitelistedIPs)): ?>
                    <p class="text-gray-400 text-center py-8">هیچ IP در لیست سفید وجود ندارد</p>
                <?php else: ?>
                    <div class="overflow-x-auto">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>IP</th>
                                    <th>توضیحات</th>
                                    <th>تاریخ اضافه</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($whitelistedIPs as $ip): ?>
                                    <tr>
                                        <td>
                                            <code class="text-sm"><?php echo htmlspecialchars($ip['ip_address']); ?></code>
                                        </td>
                                        <td><?php echo htmlspecialchars($ip['description']); ?></td>
                                        <td><?php echo date('Y/m/d H:i', strtotime($ip['created_at'])); ?></td>
                                        <td>
                                            <form method="POST" class="inline">
                                                <input type="hidden" name="action" value="remove_whitelist">
                                                <input type="hidden" name="ip" value="<?php echo htmlspecialchars($ip['ip_address']); ?>">
                                                <button type="submit" class="btn btn-sm btn-danger" 
                                                        onclick="return confirm('آیا از حذف این IP از لیست سفید اطمینان دارید؟')">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        // Auto-refresh expired blocks
        setTimeout(function() {
            location.reload();
        }, 60000); // Refresh every minute


    </script>
</body>
</html>
