<?php
/**
 * Security Email Sender - ارسال‌کننده ایمیل‌های امنیتی
 * فایل اصلی برای ارسال هشدارهای امنیتی
 */

// Production settings
error_reporting(0);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Start output buffering
ob_start();

try {
    // Clear any previous output
    ob_clean();
    
    // Get POST data
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('داده‌های ورودی نامعتبر است');
    }
    
    $recipient = $input['recipient'] ?? '';
    $action = $input['action'] ?? '';
    
    if (!$recipient) {
        throw new Exception('آدرس ایمیل گیرنده الزامی است');
    }
    
    if (!filter_var($recipient, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('آدرس ایمیل نامعتبر است');
    }
    
    switch ($action) {
        case 'test':
            $result = sendTestEmail($recipient);
            break;
            
        case 'security_alert':
            $subject = $input['subject'] ?? 'هشدار امنیتی';
            $message = $input['message'] ?? '';
            $severity = $input['severity'] ?? 'medium';
            $result = sendSecurityAlert($recipient, $subject, $message, $severity);
            break;
            
        case 'threat_detected':
            $threatType = $input['threat_type'] ?? '';
            $ipAddress = $input['ip_address'] ?? '';
            $details = $input['details'] ?? '';
            $result = sendThreatAlert($recipient, $threatType, $ipAddress, $details);
            break;
            
        default:
            throw new Exception('نوع عملیات نامعتبر است');
    }
    
    if ($result['success']) {
        echo json_encode([
            'success' => true,
            'message' => $result['message'],
            'recipient' => $recipient,
            'sent_time' => date('Y-m-d H:i:s'),
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    } else {
        throw new Exception($result['message']);
    }
    
} catch (Exception $e) {
    // Clear any output and return JSON error
    ob_clean();
    error_log("Security email error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'خطا در ارسال ایمیل: ' . $e->getMessage()
    ]);
}

// End output buffering
ob_end_flush();

/**
 * Send test email
 */
function sendTestEmail($recipient) {
    $subject = 'تست سیستم هشدار امنیتی - Gavahito';
    $message = '
    <html dir="rtl" lang="fa">
    <head>
        <meta charset="UTF-8">
        <style>
            body { font-family: "Vazir", Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #041c3f 0%, #c5912c 100%); color: white; padding: 20px; border-radius: 10px 10px 0 0; text-align: center; }
            .content { background: #f8f9fa; padding: 20px; border-radius: 0 0 10px 10px; }
            .info-box { background: rgba(4, 28, 63, 0.1); border-left: 4px solid #041c3f; padding: 15px; margin: 15px 0; }
            .success-box { background: #e8f5e8; border-left: 4px solid #4caf50; padding: 15px; margin: 15px 0; }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1>🧪 تست سیستم هشدار امنیتی</h1>
                <p>Gavahito Security System</p>
            </div>
            <div class="content">
                <div class="info-box">
                    <h3>📊 اطلاعات سیستم:</h3>
                    <ul>
                        <li><strong>زمان ارسال:</strong> ' . date('Y-m-d H:i:s') . '</li>
                        <li><strong>آدرس گیرنده:</strong> ' . $recipient . '</li>
                        <li><strong>وضعیت سیستم:</strong> فعال</li>
                        <li><strong>نسخه:</strong> 2.0.1</li>
                    </ul>
                </div>
                
                <div class="info-box">
                    <h3>🔧 تنظیمات فعلی:</h3>
                    <ul>
                        <li><strong>هشدارهای ایمیل:</strong> فعال</li>
                        <li><strong>سطح هشدار:</strong> متوسط</li>
                        <li><strong>فرکانس:</strong> فوری</li>
                    </ul>
                </div>
                
                <div class="success-box">
                    <h3>✅ تست موفق</h3>
                    <p>اگر این ایمیل را دریافت کرده‌اید، سیستم هشدار امنیتی به درستی کار می‌کند.</p>
                </div>
            </div>
        </div>
    </body>
    </html>';
    
    return sendEmail($recipient, $subject, $message);
}

/**
 * Send security alert
 */
function sendSecurityAlert($recipient, $subject, $message, $severity = 'medium') {
    $severityColors = [
        'low' => '#28a745',
        'medium' => '#ffc107',
        'high' => '#fd7e14',
        'critical' => '#dc3545'
    ];
    
    $severityText = [
        'low' => 'کم',
        'medium' => 'متوسط',
        'high' => 'بالا',
        'critical' => 'بحرانی'
    ];
    
    $color = $severityColors[$severity];
    $severityLabel = $severityText[$severity];
    
    $htmlMessage = '
    <html dir="rtl" lang="fa">
    <head>
        <meta charset="UTF-8">
        <style>
            body { font-family: "Vazir", Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #041c3f 0%, #c5912c 100%); color: white; padding: 20px; border-radius: 10px 10px 0 0; text-align: center; }
            .content { background: #f8f9fa; padding: 20px; border-radius: 0 0 10px 10px; }
            .alert { background: ' . $color . '; color: white; padding: 15px; border-radius: 5px; margin: 15px 0; }
            .info { background: rgba(4, 28, 63, 0.1); border-left: 4px solid #041c3f; padding: 10px; margin: 10px 0; }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1>🚨 هشدار امنیتی</h1>
                <p>سیستم تشخیص تهدید</p>
            </div>
            <div class="content">
                <div class="alert">
                    <h3>سطح تهدید: ' . $severityLabel . '</h3>
                    <p><strong>' . $subject . '</strong></p>
                </div>
                <div class="info">
                    ' . $message . '
                </div>
                <div style="text-align: center; margin-top: 20px; padding: 15px; background: #e9ecef; border-radius: 5px; font-size: 12px;">
                    <p>این ایمیل توسط سیستم تشخیص تهدید ارسال شده است.</p>
                    <p>زمان: ' . date('Y-m-d H:i:s') . '</p>
                </div>
            </div>
        </div>
    </body>
    </html>';
    
    return sendEmail($recipient, $subject, $htmlMessage);
}

/**
 * Send threat alert
 */
function sendThreatAlert($recipient, $threatType, $ipAddress, $details) {
    $subject = 'تشخیص تهدید - ' . $threatType;
    $message = '
    <html dir="rtl" lang="fa">
    <head>
        <meta charset="UTF-8">
        <style>
            body { font-family: "Vazir", Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #dc3545 0%, #c82333 100%); color: white; padding: 20px; border-radius: 10px 10px 0 0; text-align: center; }
            .content { background: #f8f9fa; padding: 20px; border-radius: 0 0 10px 10px; }
            .threat-box { background: #f8d7da; border-left: 4px solid #dc3545; padding: 15px; margin: 15px 0; }
            .info-box { background: rgba(4, 28, 63, 0.1); border-left: 4px solid #041c3f; padding: 10px; margin: 10px 0; }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1>⚠️ تشخیص تهدید</h1>
                <p>سیستم تشخیص تهدید Gavahito</p>
            </div>
            <div class="content">
                <div class="threat-box">
                    <h3>نوع تهدید: ' . $threatType . '</h3>
                    <p><strong>آدرس IP:</strong> ' . $ipAddress . '</p>
                </div>
                <div class="info-box">
                    <h4>جزئیات:</h4>
                    <p>' . $details . '</p>
                </div>
                <div style="text-align: center; margin-top: 20px; padding: 15px; background: #e9ecef; border-radius: 5px; font-size: 12px;">
                    <p>زمان تشخیص: ' . date('Y-m-d H:i:s') . '</p>
                </div>
            </div>
        </div>
    </body>
    </html>';
    
    return sendEmail($recipient, $subject, $message);
}

/**
 * Send email function
 */
function sendEmail($to, $subject, $message) {
    $headers = [
        'From: Gavahito Security <info@gavahito.ir>',
        'Reply-To: info@gavahito.ir',
        'Content-Type: text/html; charset=UTF-8',
        'X-Mailer: PHP/' . phpversion()
    ];
    
    $mailSent = mail($to, $subject, $message, implode("\r\n", $headers));
    
    if ($mailSent) {
        return ['success' => true, 'message' => 'ایمیل با موفقیت ارسال شد'];
    } else {
        return ['success' => false, 'message' => 'خطا در ارسال ایمیل'];
    }
}
?>
