<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session with enhanced security
if (session_status() === PHP_SESSION_NONE) {
    // Configure session security
    ini_set('session.cookie_httponly', 1);
    ini_set('session.cookie_secure', isset($_SERVER['HTTPS']));
    ini_set('session.use_strict_mode', 1);
    ini_set('session.cookie_samesite', 'Strict');
    ini_set('session.gc_maxlifetime', 1800); // 30 minutes for 2FA
    ini_set('session.cookie_lifetime', 0); // Session cookie
    
    session_start();
    
    // Regenerate session ID for 2FA page
    if (!isset($_SESSION['2fa_session_regenerated'])) {
        session_regenerate_id(true);
        $_SESSION['2fa_session_regenerated'] = true;
    }
}

require_once 'config/database.php';
require_once 'includes/functions.php';
require_once 'includes/ip-geolocation.php';
require_once 'config/email-config.php';

// Check if user is in 2FA process
if (!isset($_SESSION['2fa_user_id']) || !isset($_SESSION['2fa_email']) || !isset($_SESSION['2fa_expires'])) {
    logSecurityEvent('unauthorized_2fa_access', 'Unauthorized access to 2FA page from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit();
}

// Check if 2FA session has expired
if (time() > $_SESSION['2fa_expires']) {
    // Clear 2FA session
    unset($_SESSION['2fa_user_id']);
    unset($_SESSION['2fa_username']);
    unset($_SESSION['2fa_email']);
    unset($_SESSION['2fa_expires']);
    
    logSecurityEvent('2fa_session_expired', '2FA session expired for IP: ' . getClientIP(), 'medium');
    header('Location: 403.php');
    exit();
}

$error = '';
$success = '';
$remainingTime = $_SESSION['2fa_expires'] - time();
$remainingMinutes = floor($remainingTime / 60);
$remainingSeconds = $remainingTime % 60;



// Handle verification code submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['resend_code'])) {
    $code = preg_replace('/[^0-9]/', '', $_POST['code'] ?? '');
    
    // Enhanced server-side validation
    if (empty($code)) {
        $error = 'لطفاً کد تایید را وارد کنید';
        logSecurityEvent('2fa_empty_code', 'Empty code submitted from IP: ' . getClientIP(), 'medium');
    } elseif (strlen($code) !== 6) {
        $error = 'کد تایید باید دقیقاً 6 رقم باشد';
        logSecurityEvent('2fa_invalid_code_length', 'Invalid code length: ' . strlen($code) . ' from IP: ' . getClientIP(), 'medium');
    } elseif (!preg_match('/^[0-9]{6}$/', $code)) {
        $error = 'کد تایید فقط باید شامل اعداد باشد';
        logSecurityEvent('2fa_invalid_code_format', 'Invalid code format from IP: ' . getClientIP(), 'medium');
    } else {
        try {
            $db = new Database();
            try {
            $conn = $db->getConnection();
            } catch (Exception $connEx) {
                error_log("2FA Database connection error: " . $connEx->getMessage());
                error_log("2FA Database connection error trace: " . $connEx->getTraceAsString());
                throw new PDOException("خطا در اتصال به دیتابیس: " . $connEx->getMessage(), 0, $connEx);
            }
            
                    // Check if code is valid and not expired
            // Convert code to string to ensure proper comparison
            $codeStr = (string)$code;
            $userId = (int)$_SESSION['2fa_user_id'];
            
            // First, get all matching codes for debugging
            // Use COALESCE to handle both BOOLEAN and TINYINT types
            $query = "SELECT id, user_id, code, expires_at, is_used, created_at 
                     FROM verification_codes 
                     WHERE user_id = ? AND code = ? AND (is_used = 0 OR is_used IS NULL)
                     ORDER BY created_at DESC LIMIT 5";
            $stmt = $conn->prepare($query);
            $stmt->execute([$userId, $codeStr]);
            $allCodes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $stmt->closeCursor();
            
            // Find valid non-expired code
            $verificationCode = null;
            $currentTime = new DateTime();
            
            foreach ($allCodes as $codeRow) {
                try {
                    $expiresAt = new DateTime($codeRow['expires_at']);
                    if ($expiresAt > $currentTime) {
                        $verificationCode = $codeRow;
                        break;
                    }
                } catch (Exception $dateEx) {
                    error_log("2FA date parsing error: " . $dateEx->getMessage() . " for expires_at: " . $codeRow['expires_at']);
                    continue;
                }
            }
            
            if ($verificationCode) {
                // Mark code as used (handle both BOOLEAN and TINYINT)
                $updateQuery = "UPDATE verification_codes SET is_used = 1, used_at = NOW() WHERE id = ?";
                $updateStmt = $conn->prepare($updateQuery);
                $updateResult = $updateStmt->execute([$verificationCode['id']]);
                $updateStmt->closeCursor();
                
                if (!$updateResult) {
                    error_log("2FA Error: Failed to update verification code status for ID: " . $verificationCode['id']);
                    throw new Exception("خطا در به‌روزرسانی وضعیت کد تایید");
                }
                
                // Get user details
                $userQuery = "SELECT id, username, email FROM admin_users WHERE id = ?";
                $userStmt = $conn->prepare($userQuery);
                $userStmt->execute([$_SESSION['2fa_user_id']]);
                $user = $userStmt->fetch(PDO::FETCH_ASSOC);
                $userStmt->closeCursor();
                
                if (!$user || empty($user)) {
                    error_log("2FA Error: User not found with ID: " . $_SESSION['2fa_user_id']);
                    throw new Exception("کاربر یافت نشد. لطفاً دوباره وارد شوید.");
                }
                
                // Set session variables for logged in user
                $_SESSION['admin_id'] = $user['id'];
                $_SESSION['admin_username'] = $user['username'];
                $_SESSION['admin_role'] = 'admin';
                $_SESSION['last_activity'] = time();
                
                // Clear 2FA session variables
                unset($_SESSION['2fa_user_id']);
                unset($_SESSION['2fa_username']);
                unset($_SESSION['2fa_email']);
                unset($_SESSION['2fa_expires']);
                
                // Update last login
                $updateLoginQuery = "UPDATE admin_users SET last_login = NOW() WHERE id = ?";
                $updateLoginStmt = $conn->prepare($updateLoginQuery);
                $updateLoginStmt->execute([$user['id']]);
                $updateLoginStmt->closeCursor();
                
                // Log successful verification
                if (function_exists('logSecurityEvent')) {
                    logSecurityEvent('2fa_verification_successful', 'User: ' . $user['username'], 'low');
                }
                
                // Send login notification email
                try {
                    $clientIP = getClientIP();
                    $ipDetails = getIPDetails($clientIP);
                    $country = $ipDetails['country'] ?? 'Unknown';
                    $city = $ipDetails['city'] ?? 'Unknown';
                    $loginDate = date('Y/m/d H:i:s');
                    $loginTime = date('H:i:s');
                    $loginDatePersian = date('Y/m/d');
                    
                    $subject = '🔐 ورود موفق به پنل مدیریت گواهیتو';
                    $message = "
                    <h3 style='color: #041c3f; margin-bottom: 20px;'>✅ ورود موفق به پنل مدیریت</h3>
                    
                    <div style='background: #f8f9fa; border: 1px solid #e9ecef; border-radius: 8px; padding: 20px; margin: 20px 0;'>
                        <div style='margin-bottom: 15px;'>
                            <strong style='color: #333; display: inline-block; min-width: 120px;'>👤 کاربر:</strong>
                            <span style='color: #666;'>" . htmlspecialchars($user['username']) . "</span>
                        </div>
                        
                        <div style='margin-bottom: 15px;'>
                            <strong style='color: #333; display: inline-block; min-width: 120px;'>🌐 IP Address:</strong>
                            <span style='color: #666;'>" . htmlspecialchars($clientIP) . "</span>
                        </div>
                        
                        <div style='margin-bottom: 15px;'>
                            <strong style='color: #333; display: inline-block; min-width: 120px;'>🌍 کشور:</strong>
                            <span style='color: #666;'>" . htmlspecialchars($country) . "</span>
                        </div>
                        
                        " . (!empty($city) && $city !== 'Unknown' ? "
                        <div style='margin-bottom: 15px;'>
                            <strong style='color: #333; display: inline-block; min-width: 120px;'>🏙️ شهر:</strong>
                            <span style='color: #666;'>" . htmlspecialchars($city) . "</span>
                        </div>
                        " : "") . "
                        
                        <div style='margin-bottom: 15px;'>
                            <strong style='color: #333; display: inline-block; min-width: 120px;'>📅 تاریخ:</strong>
                            <span style='color: #666;'>" . htmlspecialchars($loginDate) . "</span>
                        </div>
                        
                        <div style='margin-bottom: 15px;'>
                            <strong style='color: #333; display: inline-block; min-width: 120px;'>🕐 زمان:</strong>
                            <span style='color: #666;'>" . htmlspecialchars($loginTime) . "</span>
                        </div>
                    </div>
                    
                    <div style='background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                        <p style='margin: 0; color: #856404;'>
                            <strong>⚠️ توجه:</strong> اگر شما این ورود را انجام نداده‌اید، لطفاً فوراً رمز عبور خود را تغییر دهید.
                        </p>
                    </div>
                    
                    <div style='background: rgba(4, 28, 63, 0.1); border: 1px solid #041c3f; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                        <p style='margin: 0; color: #041c3f;'>
                            🔒 این ایمیل به صورت خودکار برای امنیت حساب کاربری شما ارسال شده است.
                        </p>
                    </div>";
                    
                    // Send email to user
                    if (!empty($user['email'])) {
                        sendEmail($user['email'], $subject, $message);
                    }
                } catch (Exception $emailEx) {
                    // Don't block login if email fails
                    error_log("2FA login notification email error: " . $emailEx->getMessage());
                }
                
                header('Location: dashboard.php');
                exit();
            } else {
                // Enhanced error message with debugging info
                $errorMsg = 'کد تایید نامعتبر یا منقضی شده است';
                
                // Log detailed information for debugging
                if (!empty($allCodes)) {
                    $latestCode = $allCodes[0];
                    try {
                        $expiresAt = new DateTime($latestCode['expires_at']);
                        $timeDiff = $currentTime->getTimestamp() - $expiresAt->getTimestamp();
                        
                        error_log("2FA Debug - User ID: $userId, Code: $codeStr");
                        error_log("2FA Debug - Latest code expires_at: " . $latestCode['expires_at']);
                        error_log("2FA Debug - Current time: " . $currentTime->format('Y-m-d H:i:s'));
                        error_log("2FA Debug - Time difference: $timeDiff seconds");
                        error_log("2FA Debug - Code is_used: " . $latestCode['is_used']);
                        
                        if ($timeDiff > 0) {
                            $errorMsg .= ' (کد منقضی شده است - ' . round($timeDiff / 60) . ' دقیقه پیش)';
                        } elseif ($latestCode['is_used'] == 1) {
                            $errorMsg .= ' (کد قبلاً استفاده شده است)';
                        }
                    } catch (Exception $dateEx) {
                        error_log("2FA Debug - Date parsing error: " . $dateEx->getMessage());
                        $errorMsg .= ' (خطا در خواندن تاریخ انقضا)';
                    }
                } else {
                    error_log("2FA Debug - No matching code found for User ID: $userId, Code: $codeStr");
                }
                
                $error = $errorMsg;
                if (function_exists('logSecurityEvent')) {
                    logSecurityEvent('2fa_verification_failed', 'User: ' . $_SESSION['2fa_username'] . ' Code: ' . $codeStr, 'medium');
                }
                
                // Check for brute force attempts
                $failedAttempts = getFailedLoginAttempts(getClientIP());
                $maxAttempts = (int)getSystemSetting('max_2fa_attempts') ?: 5;
                
                if ($failedAttempts >= $maxAttempts) {
                    // Auto-block IP for 2FA brute force
                    autoBlockIP(getClientIP(), "تجاوز از حد مجاز تلاش‌های 2FA", 30);
                    $error = "به دلیل تلاش‌های مکرر ناموفق، IP شما برای 30 دقیقه مسدود شده است";
                    logSecurityEvent('2fa_ip_blocked', "IP blocked due to $maxAttempts failed 2FA attempts", 'high');
                }
            }
        } catch (PDOException $e) {
            error_log("2FA PDO error: " . $e->getMessage());
            error_log("2FA PDO error code: " . $e->getCode());
            error_log("2FA PDO error trace: " . $e->getTraceAsString());
            $error = 'خطا در اتصال به دیتابیس. لطفاً دوباره تلاش کنید';
            // Show more details in development (remove in production)
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                $error .= ' (' . htmlspecialchars($e->getMessage()) . ')';
            }
        } catch (Exception $e) {
            error_log("2FA verification error: " . $e->getMessage());
            error_log("2FA verification error trace: " . $e->getTraceAsString());
            $error = 'خطا در سیستم. لطفاً دوباره تلاش کنید';
            // Show more details in development (remove in production)
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                $error .= ' (' . htmlspecialchars($e->getMessage()) . ')';
            }
        }
    }
}

// Handle resend code request with enhanced security
if (isset($_POST['resend_code'])) {
    // Rate limiting for resend
    if (!checkRateLimit('resend_2fa', 3)) {
        $error = 'تعداد درخواست‌های ارسال مجدد بیش از حد مجاز است. لطفاً کمی صبر کنید';
        logSecurityEvent('2fa_resend_rate_limit', 'Resend rate limit exceeded from IP: ' . getClientIP(), 'medium');
    } else {
        try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Generate new code
        $code = sprintf('%06d', mt_rand(100000, 999999));
        $expirySeconds = (int)getSystemSetting('twofa_code_expiry') ?: 300; // Default 5 minutes
        $expiryMinutes = $expirySeconds / 60;
        
        // Use MySQL NOW() to avoid timezone issues
        $insertQuery = "INSERT INTO verification_codes (user_id, code, email, expires_at) 
                       VALUES (?, ?, ?, DATE_ADD(NOW(), INTERVAL ? SECOND))";
        $insertStmt = $conn->prepare($insertQuery);
        $insertStmt->execute([$_SESSION['2fa_user_id'], $code, $_SESSION['2fa_email'], $expirySeconds]);
        
        // Send email with new code
        require_once 'config/email-config.php';
        require_once 'config/smtp-config.php';
        $subject = 'کد تایید جدید - پنل مدیریت گواهیتو';
        $message = "
        <h3 style='color: #041c3f; margin-bottom: 20px;'>🔐 کد تایید جدید</h3>
        
        <div style='background: #f8f9fa; border: 1px solid #e9ecef; border-radius: 8px; padding: 20px; margin: 20px 0; text-align: center;'>
            <h2 style='color: #333; font-size: 2em; margin: 0; letter-spacing: 5px;'>$code</h2>
        </div>
        
        <div style='background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px; padding: 15px; margin: 20px 0;'>
            <p style='margin: 0; color: #856404;'><strong>⚠️ توجه:</strong> این کد تا $expiryMinutes دقیقه معتبر است</p>
        </div>
        
        <div style='background: rgba(4, 28, 63, 0.1); border: 1px solid #041c3f; border-radius: 8px; padding: 15px; margin: 20px 0;'>
            <p style='margin: 0; color: #041c3f;'>🕐 تاریخ ارسال: " . date('Y/m/d H:i:s') . "</p>
        </div>";
        
        // Check if PHPMailer is available
        $phpmailerPath = __DIR__ . '/vendor/phpmailer/phpmailer/src/';
        if (!file_exists($phpmailerPath . 'PHPMailer.php')) {
            error_log("PHPMailer not found at: $phpmailerPath");
            $error = 'PHPMailer نصب نشده است. لطفاً با ادمین تماس بگیرید.';
        } else {
            if (sendEmailWithFallback($_SESSION['2fa_email'], $subject, $message)) {
                // Update session expiry
                $_SESSION['2fa_expires'] = time() + ($expiryMinutes * 60);
                $success = 'کد تایید جدید ارسال شد';
                if (function_exists('logSecurityEvent')) {
                    logSecurityEvent('2fa_code_resent', 'User: ' . $_SESSION['2fa_username'], 'low');
                }
            } else {
                $error = 'خطا در ارسال کد تایید جدید. لطفاً دوباره تلاش کنید.';
            }
        }
            } catch (Exception $e) {
            error_log("2FA resend error: " . $e->getMessage());
            $error = 'خطا در ارسال کد تایید جدید';
        }
    }
}


?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تایید دو مرحله‌ای - گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
</head>
<body>
    <div class="main-content" style="max-width: 500px; margin: 2rem auto; padding: 0 1rem;">
        <div class="page-header text-center">
            <h1 class="page-title">🔐 تایید دو مرحله‌ای</h1>
            <p class="page-description">کد تایید به ایمیل شما ارسال شده است</p>
        </div>
        
        <div class="card fade-in" style="box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);">
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <div style="display: flex; align-items: center; gap: 10px;">
                        <span style="font-size: 1.2em;">⚠️</span>
                        <span><?php echo htmlspecialchars($error); ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success">
                    <div style="display: flex; align-items: center; gap: 10px;">
                        <span style="font-size: 1.2em;">✅</span>
                        <span><?php echo htmlspecialchars($success); ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <div style="background: var(--dark-700); padding: 20px; border-radius: 8px; margin-bottom: 20px; text-align: center;">
                <div style="font-size: 3em; margin-bottom: 10px;">📧</div>
                <h3 style="color: var(--primary-400); margin: 0 0 10px 0;">ایمیل تایید</h3>
                <p style="color: var(--dark-300); margin: 0; font-size: 0.9rem;">
                    کد تایید به آدرس زیر ارسال شده است:
                </p>
                <p style="color: var(--primary-400); font-weight: 600; margin: 10px 0 0 0;">
                    <?php echo htmlspecialchars($_SESSION['2fa_email']); ?>
                </p>
                

            </div>
            
            <div style="background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px; padding: 15px; margin-bottom: 20px;">
                <div style="display: flex; align-items: center; gap: 10px;">
                    <span style="font-size: 1.2em;">⏰</span>
                    <div>
                        <p style="margin: 0; color: #856404; font-weight: 600;">زمان باقی‌مانده:</p>
                        <p style="margin: 5px 0 0 0; color: #856404;" id="countdown">
                            <?php echo $remainingMinutes; ?> دقیقه و <?php echo $remainingSeconds; ?> ثانیه
                        </p>
                    </div>
                </div>
            </div>
            
            <form method="POST" action="" id="verifyForm">
                <div class="form-group">
                    <label for="code" class="form-label">
                        <span style="margin-left: 8px;">🔢</span>
                        کد تایید 6 رقمی
                    </label>
                    <input type="text" id="code" name="code" class="form-input" 
                           maxlength="6" pattern="[0-9]{6}" 
                           placeholder="000000" required 
                           style="text-align: center; font-size: 1.5em; letter-spacing: 5px; font-weight: 600;">
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn btn-primary btn-lg" style="width: 100%; padding: 15px; font-size: 1.1em; font-weight: 600;">
                        <span style="margin-left: 8px;">✅</span>
                        تایید و ورود
                    </button>
                </div>
            </form>
            
            <form method="POST" action="" style="margin-top: 20px;">
                <button type="submit" name="resend_code" class="btn btn-secondary" style="width: 100%; padding: 12px;">
                    <span style="margin-left: 8px;">🔄</span>
                    ارسال مجدد کد
                </button>
            </form>
            
            <div style="text-align: center; margin-top: 2rem; padding-top: 1rem; border-top: 1px solid var(--dark-600);">
                <a href="hf8fAS7wkd99iudu8.php" style="color: var(--primary-400); text-decoration: none; font-size: 0.9rem;">
                    ← بازگشت به صفحه ورود
                </a>
            </div>
        </div>
        
        <div style="text-align: center; margin-top: 2rem;">
            <p style="color: var(--dark-400); font-size: 0.875rem;">
                © 2024 گواهیتو Admin Panel - <a href="https://gavahito.ir/" style="color: var(--primary-400); text-decoration: none;">gavahito.ir</a>
            </p>
        </div>
    </div>
    
    <script>
        // Auto-focus on code input
        document.getElementById('code').focus();
        
        // Countdown timer
        let timeLeft = <?php echo $remainingTime; ?>;
        
        function updateCountdown() {
            if (timeLeft <= 0) {
                document.getElementById('countdown').innerHTML = 'زمان منقضی شده';
                document.getElementById('countdown').style.color = '#ef4444';
                return;
            }
            
            const minutes = Math.floor(timeLeft / 60);
            const seconds = timeLeft % 60;
            
            document.getElementById('countdown').innerHTML = 
                minutes + ' دقیقه و ' + seconds + ' ثانیه';
            
            timeLeft--;
            setTimeout(updateCountdown, 1000);
        }
        
        updateCountdown();
        
        // Auto-submit when 6 digits are entered
        document.getElementById('code').addEventListener('input', function() {
            const code = this.value.replace(/\D/g, ''); // Remove non-digits
            this.value = code;
            
            if (code.length === 6) {
                document.getElementById('verifyForm').submit();
            }
        });
        
        // Enhanced security validation for verification form
        document.getElementById('verifyForm').addEventListener('submit', function(e) {
            const code = document.getElementById('code').value.trim();
            const submitButton = this.querySelector('button[type="submit"]');
            
            // Server-side validation (this is secure)
            if (!code || code.length !== 6 || !/^\d{6}$/.test(code)) {
                e.preventDefault();
                showNotification('کد تایید باید دقیقاً 6 رقم باشد', 'error');
                return false;
            }
            
            // XSS prevention (this is secure)
            if (code.includes('<') || code.includes('>') || code.includes('"') || code.includes("'") || code.includes('&')) {
                e.preventDefault();
                showNotification('کد تایید شامل کاراکترهای غیرمجاز است', 'error');
                return false;
            }
            
            // Show loading state
            submitButton.disabled = true;
            submitButton.innerHTML = '<span style="margin-left: 8px;">⏳</span>در حال تایید...';
        });
        

        
        // Show notification function
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 15px 20px;
                border-radius: 8px;
                color: white;
                font-weight: 600;
                z-index: 10000;
                animation: slideIn 0.3s ease-out;
                max-width: 300px;
            `;
            
            if (type === 'error') {
                notification.style.background = '#ef4444';
            } else if (type === 'success') {
                notification.style.background = '#10b981';
            } else {
                notification.style.background = '#3b82f6';
            }
            
            notification.textContent = message;
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.animation = 'slideOut 0.3s ease-in';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }
        
        // Add CSS animations
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideIn {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOut {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
            
            @media (max-width: 768px) {
                .main-content {
                    margin: 1rem auto !important;
                    padding: 0 0.5rem !important;
                }
                .card {
                    padding: 1.5rem !important;
                }
                .form-input {
                    font-size: 16px !important;
                }
            }
        `;
        document.head.appendChild(style);
        
        // Check if page is loaded after form submission (back button or refresh)
        window.addEventListener('load', function() {
            const submitButton = document.querySelector('button[type="submit"]');
            if (submitButton && submitButton.disabled) {
                // Reset button if it's still in loading state
                submitButton.disabled = false;
                submitButton.innerHTML = '<span style="margin-left: 8px;">✅</span>تایید و ورود';
            }
        });
    </script>
</body>
</html>
